
###PLOT COM PHYLOMORPHOSPACE

#FILE: Macaulay_BirdCoM_Phylomorphospace_CoM.R

#CITATION: Macaulay, S. et al. XXXX. Decoupling body shape and mass distribution in birds and their dinosaurian ancestors

#CODE CREDITS: Built by Samuel Cross in R. 4.1.2

#DESCRIPTION: This code plots Figure 2b from Macaulay et al., the normalised craniocaudal and dorsoventral centre of mass 
#             values, presented as a phylomorphospace, using the Phytools FastANC function to infer ancestral nodes. The
#             code is split into three sections, comprising the main phylomorphospace and the two sub-figures. The code
#             does not plot these together (or format the sub-figures), as these were manually combined for the final figure.

###Load Dependent Packages

library(ape)
library(phytools)
library(plotrix)
library(geiger)
library(xlsx)

###Load Data

Data <- read.xlsx("Macaulay_BirdCoM_PCM_Dataset.xlsx", sheetIndex = 2, row.names = 1)

###Organise Taxon Lists

FLD_list <- row.names(Data)[Data[,1] == 'Forelimb dominated']
HLD_list <- row.names(Data)[Data[,1] == 'Hind limb dominated']

###Load Tree

tree <- read.tree("Macaulay_BirdCoM_PCM_Tree.tre") #Change to use other trees/time-scaling methods

###Paint Tree for Plotting

tree<-paintSubTree(tree,node=94,state="2")
tree<-paintSubTree(tree,node=87,state="2")
tree<-paintSubTree(tree,node=67,state="3")
tree<-paintBranches(tree,edge=87,state="2")
tree<-paintBranches(tree,edge=29,state="3")
tree<-paintBranches(tree,edge=19,state="2")
tree<-paintBranches(tree,edge=52,state="4")
tree<-paintBranches(tree,edge=53,state="4")
tree<-paintBranches(tree,edge=98,state="4")
tree<-paintBranches(tree,edge=99,state="4")
tree<-paintBranches(tree,edge=50,state="4")
tree<-paintBranches(tree,edge=48,state="4")
tree<-paintBranches(tree,edge=49,state="4")
tree<-paintBranches(tree,edge=47,state="4")

cols<-c("black","red","blue","green")
names(cols)<-1:4
plotSimmap(tree,cols,pts=FALSE)

###Write Convex Hull Function

Plot_ConvexHull<-function(xcoord, ycoord, bgcolor){
  hpts <- chull(x = xcoord, y = ycoord)
  hpts <- c(hpts, hpts[1])
  #lines(xcoord[hpts], ycoord[hpts], col = lcolor)
  polygon(x = xcoord[hpts], y =  ycoord[hpts], col = adjustcolor(bgcolor, alpha.f = 0.2) , border = NA)
}

###Export Phylomorphospace Figure

tiff(filename = "CoM_Phylomorphospace.tif", compression = "lzw", width = 222, height = 166, units = "mm", res = 600) #166x166*2 or 166x166*1.5

plot.new()

#Generate Phylomorphospace object using FastANC to estimate internal node values
Phylomorph <- phylomorphospace(tree,Data[,4:3],
                               colors=cols,
                               bty="o",
                               node.by.map=TRUE,
                               node.size = c(1, 1.2),
                               axes = FALSE,
                               ylim = c(-40, 0),
                               xlim = c(0, 160),
                               xlab ="", 
                               ylab ="Dorsoventral CoM",
                               label = "off")

#Plot own axes so X is on top
axis(3)
mtext("Craniocaudal CoM", side = 3, line = 3)
axis(2)

#Plot hulls around bird locomotor categories
Plot_ConvexHull(xcoord=Data[FLD_list,][,4],
                ycoord=Data[FLD_list,][,3], bgcolor="blue")
Plot_ConvexHull(xcoord=Data[HLD_list,][,4],
                ycoord=Data[HLD_list,][,3], bgcolor="red")

#Overwrite extinct taxa with squares and reconstructed nodes with yellow
points(Phylomorph$xx[34:46], Phylomorph$yy[34:46], cex = 1.4, pch = 15, col = "black")
points(Phylomorph$xx[49], Phylomorph$yy[49], cex = 1.4, pch = 22, col = "black", bg = "green")
points(Phylomorph$xx[51:66], Phylomorph$yy[51:66], cex = 1.5, pch = 21, col = "black", bg = "yellow")
points(Phylomorph$xx[86], Phylomorph$yy[86], cex = 1.5, pch = 21, col = "black", bg = "yellow")
points(Phylomorph$xx[94], Phylomorph$yy[94], cex = 1.5, pch = 21, col = "black", bg = "yellow")

#Unquote these lines to add numbers to the ancestral nodes. I added these manually for the figure however.
#text(Phylomorph$xx[51], Phylomorph$yy[51], labels = "1", cex=0.7, pos=3)
#text(Phylomorph$xx[52], Phylomorph$yy[52], labels = "2", cex=0.7, pos=3)
#text(Phylomorph$xx[55], Phylomorph$yy[55], labels = "3-5", cex=0.7, pos=2)
#text(Phylomorph$xx[56], Phylomorph$yy[56], labels = "6", cex=0.7, pos=4)
#text(Phylomorph$xx[57], Phylomorph$yy[57], labels = "7", cex=0.7, pos=3)
#text(Phylomorph$xx[58], Phylomorph$yy[58], labels = "8", cex=0.7, pos=4)
#text(Phylomorph$xx[59], Phylomorph$yy[59], labels = "9", cex=0.7, pos=4)
#text(Phylomorph$xx[60], Phylomorph$yy[60], labels = "10", cex=0.7, pos=4)
#text(Phylomorph$xx[62], Phylomorph$yy[62], labels = "11-12", cex=0.7, pos=2)
#text(Phylomorph$xx[63], Phylomorph$yy[63], labels = "13", cex=0.7, pos=2)
#text(Phylomorph$xx[64], Phylomorph$yy[64], labels = "14", cex=0.7, pos=4)
#text(Phylomorph$xx[65], Phylomorph$yy[65], labels = "15", cex=0.7, pos=1)
#text(Phylomorph$xx[66], Phylomorph$yy[66], labels = "17" ,cex=0.7, pos=1)
#text(Phylomorph$xx[94], Phylomorph$yy[94], labels = "16" ,cex=0.7, pos=1)

dev.off()

###Prepare Data for plotting subfigure 1 (Figure 2D)

#Run FastANC to get Confidence Intervals
name.check(tree,rownames(Data))
Data <- Data[match(tree$tip.label,rownames(Data)),]
Data.CC <- Data[, 'CCnorm']
names(Data.CC) <- rownames(Data)
Data.CC.ANC <- fastAnc(tree, Data.CC, vars=T, CI=T)

#Format CIs for plotting
CI.UP <- Data.CC.ANC[["CI95"]][,2] #Grab upper CIs
CI.UP <- CI.UP[-c(16:49)] #Grab extinct nodes
CI.UP <- CI.UP[c(10:13)] #Grab coelurosaurs
CI.UP <- CI.UP[-c(2)] #Remove oviraptorans as they overlap Dromaeosaurs
CI.LO <- Data.CC.ANC[["CI95"]][,1] #Grab lower CIs
CI.LO <- CI.LO[-c(16:49)] #Grab extinct nodes
CI.LO <- CI.LO[c(10:13)] #Grab coelurosaurs
CI.LO <- CI.LO[-c(2)] #Remove oviraptorans as they overlap Dromaeosaurs

#Extract nodes for plotting
CI.Nodes.X <- Phylomorph$xx[60:63] #Grab coelurosaur X values
CI.Nodes.X <- CI.Nodes.X[-c(2)] #remove oviraptorans
CI.Nodes.Y <- Phylomorph$yy[60:63] #Grab coelurosaur Y Values
CI.Nodes.Y <- CI.Nodes.Y[-c(2)] #remove oviraptorans

#Extract lines for plotting
CI.Lines.X <- Phylomorph$xx[59:64]
CI.Lines.Y <- Phylomorph$yy[59:64]

###Export Subfigure 1 - Note that this will require further prep in a graphics software

tiff(filename = "CoM_Phylomorphospace_Theropod_CIs.tif", compression = "lzw", width = 222, height = 166, units = "mm", res = 600) #166x166*2 or 166x166*1.5

plot.new()

#plot CIs
plotCI(CI.Nodes.X, CI.Nodes.Y, ui = CI.UP, li = CI.LO, err = "x", ylim = c(-40,0), xlim = c(0, 160), lwd = 2.0)

#plot FLD & HLD hull regions
Plot_ConvexHull(xcoord=Data[FLD_list,][,4],
                ycoord=Data[FLD_list,][,3], bgcolor="blue")
Plot_ConvexHull(xcoord=Data[HLD_list,][,4],
                ycoord=Data[HLD_list,][,3], bgcolor="red")

#Plot phylomorphospace lines
lines(CI.Lines.X, CI.Lines.Y, lwd = 3)

#Plot ancestral nodes
points(Phylomorph$xx[51:65], Phylomorph$yy[51:65], cex = 1.5, pch = 21, col = "black", bg = "yellow")

dev.off()

###Prepare Data for plotting subfigure 2 (Figure 2C)

#Grab CoM position & CIs for Theropoda reconstructed node
Theropoda.X <- Phylomorph$xx[55]
Theropoda.Y <- Phylomorph$yy[55]
Theropoda.CI.LO <- Data.CC.ANC[["CI95"]][5,1]
Theropoda.CI.UP <- Data.CC.ANC[["CI95"]][5,2]

#Grab CoM position & CIs for caudal-most HLD bird
HLD.Caudal.X <- Phylomorph$xx[96]
HLD.Caudal.Y <- Phylomorph$yy[96]
HLD.Caudal.CI.LO <- Data.CC.ANC[["CI95"]][46,1]
HLD.Caudal.CI.UP <- Data.CC.ANC[["CI95"]][46,2]

###Export Subfigure 2 - Note that this will require further prep in a graphics software

tiff(filename = "CoM_Phylomorphospace_MostCaudal_CIs.tif", compression = "lzw", width = 222, height = 166, units = "mm", res = 600) #166x166*2 or 166x166*1.5

plot.new()

plotCI(Theropoda.X,Theropoda.Y, ui = Theropoda.CI.UP, li = Theropoda.CI.LO, 
       err = "x", lwd = 2.0, ylim = c(-40,0), xlim = c(0, 160))

plotCI(HLD.Caudal.X,HLD.Caudal.Y, ui = HLD.Caudal.CI.UP, li = HLD.Caudal.CI.LO, 
       err = "x", ylim = c(-40,0), xlim = c(0, 160), lwd = 2.0, add = TRUE,  cex = 0.5)

points(Phylomorph$xx[96], Phylomorph$yy[96], cex = 1.5, pch = 21, col = "black", bg = "red")
points(Phylomorph$xx[55], Phylomorph$yy[55], cex = 1.5, pch = 21, col = "black", bg = "black")

dev.off()

