
###Principal Component Analysis with Branches

#FILE: Macaulay_BirdCoM_PhylomorphPCA.R

#CITATION: Macaulay, S. et al. XXXX. Decoupling body shape and mass distribution in birds and their dinosaurian ancestors

#CODE CREDITS: Built by Samuel Cross in R. 4.1.2

#DESCRIPTION: This code will perform a 'normal' PCA analysis using FactoMineR, it will then take the coordinates and
#             use the phylomorphospace function of Phytools to estimate ancestral nodes and add branches. The code
#             will also add biplots to visualise the loadings (though these were manually repositioned for the final
#             figure).


###Load Dependent Packages

library(ape)
library(phytools)
library(plotrix)
library(geiger)
library(xlsx)
library(FactoMineR)

###Load and Prepare Tree

tree <- read.tree("Macaulay_BirdCoM_PCM_Tree.tre")
tax.list.to.go <- c("Iguana_iguana", "Melanosuchus_niger", "Crocodylus_johnstoni")
tree <- drop.tip(tree, tree$tip.label[match(tax.list.to.go, tree$tip.label)])

tree<-paintSubTree(tree,node=90,state="2")
tree<-paintSubTree(tree,node=82,state="2")
tree<-paintSubTree(tree,node=63,state="3")
tree<-paintBranches(tree,edge=19,state="2")
tree<-paintBranches(tree,edge=29,state="3")

cols<-c("black","red","blue")
names(cols)<-1:3
plotSimmap(tree,cols,pts=FALSE)

###Load and Prepare Data (and organise lists)

#Segment masses
Data.Het <- read.xlsx("Macaulay_BirdCoM_PCM_Dataset.xlsx", sheetIndex = 2, row.names = 1)
FLD_list <- row.names(Data.Het)[Data.Het[,1] == 'Forelimb dominated']
HLD_list <- row.names(Data.Het)[Data.Het[,1] == 'Hind limb dominated']
Data.Het[Data.Het == "NA"] <- 0.001
Data.Het <- Data.Het[!(row.names(Data.Het) %in% tax.list.to.go), ]
Data.Het<-Data.Het[,c(3:7,15:17)] #Select only relevant variables
Data.Het[[8]] <- as.numeric(Data.Het[[8]]) #read.xlsx does not allow "NA" specification and imports column as "character" class, this line converts back to "numeric"
name.check(tree,Data.Het)

#Rename for plotting biplots
colnames(Data.Het)[1:8] <- c("DV", "CC", "HD", "NK", "TO", "HL", "FL", "TL")

#Segment lengths
Data.Lin <- read.xlsx("Macaulay_BirdCoM_PCM_Dataset.xlsx", sheetIndex = 3, row.names = 1)
#FLD_list <- row.names(Data.Lin)[Data.Lin[,1] == 'Forelimb dominated']
#HLD_list <- row.names(Data.Lin)[Data.Lin[,1] == 'Hind limb dominated']
Data.Lin[Data.Lin == "NA"] <- 0.001
Data.Lin <- Data.Lin[!(row.names(Data.Lin) %in% tax.list.to.go), ]
Data.Lin<-Data.Lin[,c(3:4,7:15)] #Select only relevant variables
name.check(tree,Data.Lin)

#Rename for plotting biplots
colnames(Data.Lin)[1:11] <- c("DV", "CC", "PW", "PL", "SH", "NK", "HD", "HL", "FL", "GA", "TL")

###Load Convex Hull Function

Plot_ConvexHull<-function(xcoord, ycoord, bgcolor){
  hpts <- chull(x = xcoord, y = ycoord)
  hpts <- c(hpts, hpts[1])
  #lines(xcoord[hpts], ycoord[hpts], col = lcolor)
  polygon(x = xcoord[hpts], y =  ycoord[hpts], col = adjustcolor(bgcolor, alpha.f = 0.2) , border = NA)
}

###Principal Component Analysis

#Segment masses
res.pca <- PCA(Data.Het, scale.unit = TRUE, ncp = 8)
Het.scores <- res.pca$ind$coord
Het.loadings.cor <- res.pca$var$cor
Het.loadings.contrib <- res.pca$var$contrib
Het.eig <- res.pca$eig

#Segment lengths
res.pca <- PCA(Data.Lin, scale.unit = TRUE, ncp = 11)
Lin.scores <- res.pca$ind$coord
Lin.loadings.cor <- res.pca$var$cor
Lin.loadings.contrib <- res.pca$var$contrib
Lin.eig <- res.pca$eig

#Export Results to Spreadsheet
write.xlsx(Het.scores, "Normal_PCA_Results.xlsx", sheetName = "Scores_Hetero_Norm_DV&CC", col.names = T, row.names = T)
write.xlsx(Het.loadings.cor, "Normal_PCA_Results.xlsx", sheetName = "Loadings_Correlations_Hetero_Norm_DV&CC", col.names = T, row.names = T, append = T)
write.xlsx(Het.loadings.contrib, "Normal_PCA_Results.xlsx", sheetName = "Loadings_Contributions_Hetero_Norm_DV&CC", col.names = T, row.names = T, append = T)
write.xlsx(Het.eig, "Normal_PCA_Results.xlsx", sheetName = "Eigenvalues_Hetero_Norm_DV&CC", col.names = T, row.names = T, append = T)
write.xlsx(Lin.scores, "Normal_PCA_Results.xlsx", sheetName = "Scores_linear_Norm_DV&CC", col.names = T, row.names = T, append = T)
write.xlsx(Lin.loadings.cor, "Normal_PCA_Results.xlsx", sheetName = "Loadings_Correlations_linear_Norm_DV&CC", col.names = T, row.names = T, append = T)
write.xlsx(Lin.loadings.contrib, "Normal_PCA_Results.xlsx", sheetName = "Loadings_Contributions_linear_Norm_DV&CC", col.names = T, row.names = T, append = T)
write.xlsx(Lin.eig, "Normal_PCA_Results.xlsx", sheetName = "Eigenvalues_linear_Norm_DV&CC", col.names = T, row.names = T, append = T)

### Plot 4x4 Figure of PCA Phylomorphospace

tiff(filename = "PhyloPCA_4x4.tif", compression = "lzw", width = 222, height = 166, units = "mm", res = 600) #166x166*2 or 166x166*1.5

plot.new()

par(mfrow=c(2,2), mai=c(0.5,0.8,0.5,0.5))

Phylomorph <- phylomorphospace(tree,Het.scores[,1:2],
                               colors=cols,
                               bty="o",
                               node.by.map=TRUE,
                               node.size = c(1, 1.2),
                               xlab="PC1 - 36.4%",
                               ylab="PC2 - 24.9%",
                               label = "off")

Plot_ConvexHull(xcoord=Het.scores[FLD_list,][,1],
                ycoord=Het.scores[FLD_list,][,2], bgcolor="blue")
Plot_ConvexHull(xcoord=Het.scores[HLD_list,][,1],
                ycoord=Het.scores[HLD_list,][,2], bgcolor="red")

points(Phylomorph$xx[48:62], Phylomorph$yy[48:62], cex = 1.5, pch = 21, col = "black", bg = "yellow")
points(Phylomorph$xx[90], Phylomorph$yy[90], cex = 1.5, pch = 21, col = "black", bg = "yellow")
points(Phylomorph$xx[82], Phylomorph$yy[82], cex = 1.5, pch = 21, col = "black", bg = "yellow")

##

Phylomorph <- phylomorphospace(tree,Lin.scores[,1:2],
                               colors=cols,
                               bty="o",
                               node.by.map=TRUE,
                               node.size = c(1, 1.2),
                               xlab="PC1 - 37.8%",
                               ylab="PC2 - 17.7%", 
                               label = "off")

Plot_ConvexHull(xcoord=Lin.scores[FLD_list,][,1],
                ycoord=Lin.scores[FLD_list,][,2], bgcolor="blue")
Plot_ConvexHull(xcoord=Lin.scores[HLD_list,][,1],
                ycoord=Lin.scores[HLD_list,][,2], bgcolor="red")

points(Phylomorph$xx[48:62], Phylomorph$yy[48:62], cex = 1.5, pch = 21, col = "black", bg = "yellow")
points(Phylomorph$xx[90], Phylomorph$yy[90], cex = 1.5, pch = 21, col = "black", bg = "yellow")
points(Phylomorph$xx[82], Phylomorph$yy[82], cex = 1.5, pch = 21, col = "black", bg = "yellow")

##

Phylomorph <- phylomorphospace(tree,Het.scores[, c(1,3)],
                               colors=cols,
                               bty="o",
                               node.by.map=TRUE,
                               node.size = c(1, 1.2),
                               xlab="PC1 - 36.4%",
                               ylab="PC3 - 15.2%", 
                               label = "off")

Plot_ConvexHull(xcoord=Het.scores[FLD_list,][,1],
                ycoord=Het.scores[FLD_list,][,3], bgcolor="blue")
Plot_ConvexHull(xcoord=Het.scores[HLD_list,][,1],
                ycoord=Het.scores[HLD_list,][,3], bgcolor="red")

points(Phylomorph$xx[48:62], Phylomorph$yy[48:62], cex = 1.5, pch = 21, col = "black", bg = "yellow")
points(Phylomorph$xx[90], Phylomorph$yy[90], cex = 1.5, pch = 21, col = "black", bg = "yellow")
points(Phylomorph$xx[82], Phylomorph$yy[82], cex = 1.5, pch = 21, col = "black", bg = "yellow")

##

Phylomorph <- phylomorphospace(tree,Lin.scores[, c(1,3)],
                               colors=cols,
                               bty="o",
                               node.by.map=TRUE,
                               node.size = c(1, 1.2),
                               xlab="PC1 - 37.8%",
                               ylab="PC3 - 16.3%", 
                               label = "off")

Plot_ConvexHull(xcoord=Lin.scores[FLD_list,][,1],
                ycoord=Lin.scores[FLD_list,][,3], bgcolor="blue")
Plot_ConvexHull(xcoord=Lin.scores[HLD_list,][,1],
                ycoord=Lin.scores[HLD_list,][,3], bgcolor="red")

points(Phylomorph$xx[48:62], Phylomorph$yy[48:62], cex = 1.5, pch = 21, col = "black", bg = "yellow")
points(Phylomorph$xx[90], Phylomorph$yy[90], cex = 1.5, pch = 21, col = "black", bg = "yellow")
points(Phylomorph$xx[82], Phylomorph$yy[82], cex = 1.5, pch = 21, col = "black", bg = "yellow")

### Biplots

#create overlay panels with the same layout
par(mfrow=c(2,2), mai=c(0.5,0.8,0.5,0.5), new = TRUE)

##

par(mfg=c(1,1), mai=c(1.85,0.8,0.3,2)) #Note:mai is set to maintain identical aspect ratio to main plots

plot(Het.scores[,1:2], bty = 'n', yaxt="n", xaxt="n", xlab = '', ylab = '', col = alpha(cols, 0), xlim = c(-1.5,1.5), ylim = c(-1.5,1.5) )

arrows(x0 = 0, x1 = Het.loadings.cor[,1], 
       y0 = 0, y1 = Het.loadings.cor[,2], 
       col = "deeppink", 
       length = 0.08, 
       lwd = 1.5,
       angle = 30)

text(x = Het.loadings.cor[,1], y = Het.loadings.cor[,2], 
     labels = row.names(Het.loadings.cor), 
     cex = 0.8,
     font = 1,
     col = "gray50",
     pos = c(2, 4, 2, 2, 4, 4, 4, 2)) #organised according to column order

##

par(mfg=c(1,2), mai=c(1.85,0.8,0.3,2))
#par(fig = c(0,0.6,0,0.6), xpd = NA, new = TRUE)

plot(Lin.scores[,1:2], bty = 'n', yaxt="n", xaxt="n", xlab = '', ylab = '', col = alpha(cols, 0), xlim = c(-1.5,1.5), ylim = c(-1.5,1.5) )

arrows(x0 = 0, x1 = Lin.loadings.cor[,1], 
       y0 = 0, y1 = Lin.loadings.cor[,2], 
       col = "deeppink", 
       length = 0.08, 
       lwd = 1.5,
       angle = 30)

text(x = Lin.loadings.cor[,1], y = Lin.loadings.cor[,2], 
     labels = row.names(Lin.loadings.cor), 
     cex = 0.8,
     font = 1,
     col = "gray50",
     pos = c(2, 4, 1, 3, 4, 4, 3, 1, 4, 3, 2)) #organised according to column order

##

par(mfg=c(2,1), mai=c(1.85,0.8,0.3,2))

plot(Het.scores[,1:3], bty = 'n', yaxt="n", xaxt="n", xlab = '', ylab = '', col = alpha(cols, 0), xlim = c(-1.5,1.5), ylim = c(-1.5,1.5) )

arrows(x0 = 0, x1 = Het.loadings.cor[,1], 
       y0 = 0, y1 = Het.loadings.cor[,3], 
       col = "deeppink", 
       length = 0.08, 
       lwd = 1.5,
       angle = 30)

text(x = Het.loadings.cor[,1], y = Het.loadings.cor[,3], 
     labels = row.names(Het.loadings.cor), 
     cex = 0.8,
     font = 1,
     col = "gray50",
     pos = c(2, 4, 4, 2, 4, 3, 4, 2)) #organised according to column order

##

par(mfg=c(2,2), mai=c(1.85,0.8,0.3,2))

plot(Lin.scores[,1:3], bty = 'n', yaxt="n", xaxt="n", xlab = '', ylab = '', col = alpha(cols, 0), xlim = c(-1.5,1.5), ylim = c(-1.5,1.5) )

arrows(x0 = 0, x1 = Lin.loadings.cor[,1], 
       y0 = 0, y1 = Lin.loadings.cor[,3], 
       col = "deeppink", 
       length = 0.08, 
       lwd = 1.5,
       angle = 30)

text(x = Lin.loadings.cor[,1], y = Lin.loadings.cor[,3], 
     labels = row.names(Lin.loadings.cor), 
     cex = 0.8,
     font = 1,
     col = "gray50",
     pos = c(2, 3, 2, 1, 4, 4, 3, 1, 4, 3, 2)) #organised according to column order

dev.off()



