
###Principal Component Analysis with Branches

#FILE: Macaulay_BirdCoM_Multiple_Phylomorphospaces.R

#CITATION: Macaulay, S. et al. XXXX. Decoupling body shape and mass distribution in birds and their dinosaurian ancestors

#CODE CREDITS: Built by Samuel Cross in R. 4.1.2

#DESCRIPTION: This code will reproduce the individual elements of Figure 2, though they were assembled manually for the
#             final figure. This includes, normalised hind- versus forelimb lengths (full-sized and zoomed), normalised
#             femur versus MT length (full-sized and zoomed), and femoral length versus craniocaudal centre of mass.

###Load Dependent Packages

library(ape)
library(phytools)
library(plotrix)
library(geiger)
library(xlsx)
library(plotrix)

###Load Data

Data <- read.xlsx("Macaulay_BirdCoM_PCM_Dataset.xlsx", sheetIndex = 3, row.names = 1)

###Organise Taxon Lists

FLD_list <- row.names(Data)[Data[,1] == 'Forelimb dominated']
HLD_list <- row.names(Data)[Data[,1] == 'Hind limb dominated']

###Load Tree

tree <- read.tree("Macaulay_BirdCoM_PCM_Tree.tre") #Change to use other trees/time-scaling methods

###Paint Tree for Plotting

tree<-paintSubTree(tree,node=94,state="2")
tree<-paintSubTree(tree,node=87,state="2")
tree<-paintSubTree(tree,node=67,state="3")
tree<-paintBranches(tree,edge=87,state="2")
tree<-paintBranches(tree,edge=29,state="3")
tree<-paintBranches(tree,edge=19,state="2")
tree<-paintBranches(tree,edge=52,state="4")
tree<-paintBranches(tree,edge=53,state="4")
tree<-paintBranches(tree,edge=98,state="4")
tree<-paintBranches(tree,edge=99,state="4")
tree<-paintBranches(tree,edge=50,state="4")
tree<-paintBranches(tree,edge=48,state="4")
tree<-paintBranches(tree,edge=49,state="4")
tree<-paintBranches(tree,edge=47,state="4")

cols<-c("black","red","blue","green")
names(cols)<-1:4
plotSimmap(tree,cols,pts=FALSE)

###Write Convex Hull Function

Plot_ConvexHull<-function(xcoord, ycoord, bgcolor){
  hpts <- chull(x = xcoord, y = ycoord)
  hpts <- c(hpts, hpts[1])
  #lines(xcoord[hpts], ycoord[hpts], col = lcolor)
  polygon(x = xcoord[hpts], y =  ycoord[hpts], col = adjustcolor(bgcolor, alpha.f = 0.2) , border = NA)
}


###Export Combined Phylomorphospace Plot of Linear Parametres in Figures 3A-D

tiff(filename = "Combined_Phylomorphospace.tif", compression = "lzw", width = 222, height = 222, units = "mm", res = 600) #166x166*2 or 166x166*1.5

plot.new()

par(mfrow=c(2,2), mai=c(0.5,0.5,0.5,0.5))

#Fore versus Hind
Phylomorph <- phylomorphospace(tree,Data[,13:12],
                               colors=cols,
                               bty="o",
                               node.by.map=TRUE,
                               node.size = c(1, 0.8),
                               axes = FALSE,
                               ylim = rev(range(Data[,12])),
                               #xlim = c(0, 160),
                               xlab ="", 
                               ylab ="", #Normalised Hindlimb Length
                               label = "off")

axis(side=3, at=seq(0,600,200))
#mtext("Normalised Forelimb Length", side = 3, line = 3)
axis(2)

Plot_ConvexHull(xcoord=Data[FLD_list,][,13],
                ycoord=Data[FLD_list,][,12], bgcolor="blue")
Plot_ConvexHull(xcoord=Data[HLD_list,][,13],
                ycoord=Data[HLD_list,][,12], bgcolor="red")

#Plotting order: Minor Internal nodes -> Taxon Values -> Major Ancestral Nodes
points(Phylomorph$xx[67:85], Phylomorph$yy[67:85], cex = 1.3, pch = 21, col = "black", bg = "blue")
points(Phylomorph$xx[87:93], Phylomorph$yy[87:93], cex = 1.3, pch = 21, col = "black", bg = "red")
points(Phylomorph$xx[95:96], Phylomorph$yy[95:96], cex = 1.3, pch = 21, col = "black", bg = "red")
points(Phylomorph$xx[97], Phylomorph$yy[97], cex = 1.3, pch = 21, col = "black", bg = "black")
points(Phylomorph$xx[98:99], Phylomorph$yy[98:99], cex = 1.3, pch = 21, col = "black", bg = "green")
points(Phylomorph$xx[1:18], Phylomorph$yy[1:18], cex = 1.5, pch = 21, col = "black", bg = "blue")
points(Phylomorph$xx[19], Phylomorph$yy[19], cex = 1.5, pch = 21, col = "black", bg = "red")
points(Phylomorph$xx[20], Phylomorph$yy[20], cex = 1.5, pch = 21, col = "black", bg = "blue")
points(Phylomorph$xx[21:28], Phylomorph$yy[21:28], cex = 1.5, pch = 21, col = "black", bg = "red")
points(Phylomorph$xx[29], Phylomorph$yy[29], cex = 1.5, pch = 21, col = "black", bg = "blue")
points(Phylomorph$xx[30:33], Phylomorph$yy[30:33], cex = 1.5, pch = 21, col = "black", bg = "red")
points(Phylomorph$xx[34:46], Phylomorph$yy[34:46], cex = 1.5, pch = 15, col = "black")
points(Phylomorph$xx[47:48], Phylomorph$yy[47:48], cex = 1.5, pch = 21, col = "black", bg = "green")
points(Phylomorph$xx[49], Phylomorph$yy[49], cex = 1.5, pch = 22, col = "black", bg = "green")
points(Phylomorph$xx[50], Phylomorph$yy[50], cex = 1.5, pch = 21, col = "black", bg = "green")
points(Phylomorph$xx[51:66], Phylomorph$yy[51:66], cex = 1.6, pch = 21, col = "black", bg = "yellow")
points(Phylomorph$xx[86], Phylomorph$yy[86], cex = 1.6, pch = 21, col = "black", bg = "yellow")
points(Phylomorph$xx[94], Phylomorph$yy[94], cex = 1.6, pch = 21, col = "black", bg = "yellow")

#Fore versus Hind Zoomed
Phylomorph <- phylomorphospace(tree,Data[,13:12],
                               colors=cols,
                               bty="o",
                               node.by.map=TRUE,
                               node.size = c(1, 0.8),
                               axes = FALSE,
                               ylim = c(350, 150),
                               xlim = c(50, 350),
                               xlab ="", 
                               ylab ="", #Normalised Hindlimb Length
                               label = "off")

axis(3, at=seq(50,350,100))
#mtext("Normalised Forelimb Length", side = 3, line = 3)
axis(2)

#Note: Polygon does not work when alpha values are given *and* points lie outside the plot.
#This is the case for the normal convex hull code when the plots are zoomed in.
#I have used this temporary workaround, manually exporting the coordinates of the points that form the hull, and then
#using non-alpha hulls + the locator() command to estimate where the hulls meet the zoomed-in border. I then combine
#together as a string of coordinates and use the base polygon command to manually draw the modified hull.
xpoint = c(224.3935, 128.7259, 194.9416,264.8, 363, 363, 224.3935)
ypoint = c(185.2582, 166.8025, 281.1434, 360, 360, 227.9, 185.2582)
polygon(x = xpoint, y = ypoint, col = adjustcolor("blue", alpha.f = 0.2) , border = NA)

xpoint = c(227.82214, 227.63561, 207.12948, 140.46973, 63.52939, 101.2601, 220.5196, 227.82214) 
ypoint = c(265.6174, 260.6435, 219.9006, 240.8565, 336.5668, 357.9514, 358.5592, 265.6174) 
polygon(x = xpoint, y = ypoint, col = adjustcolor("red", alpha.f = 0.2) , border = NA)

#Plotting order: Minor Internal nodes -> Taxon Values -> Major Ancestral Nodes
points(Phylomorph$xx[67:85], Phylomorph$yy[67:85], cex = 1.6, pch = 21, col = "black", bg = "blue")
points(Phylomorph$xx[87:93], Phylomorph$yy[87:93], cex = 1.6, pch = 21, col = "black", bg = "red")
points(Phylomorph$xx[95:96], Phylomorph$yy[95:96], cex = 1.6, pch = 21, col = "black", bg = "red")
points(Phylomorph$xx[97], Phylomorph$yy[97], cex = 1.6, pch = 21, col = "black", bg = "black")
points(Phylomorph$xx[98:99], Phylomorph$yy[98:99], cex = 1.6, pch = 21, col = "black", bg = "green")
points(Phylomorph$xx[1:18], Phylomorph$yy[1:18], cex = 1.8, pch = 21, col = "black", bg = "blue")
points(Phylomorph$xx[19], Phylomorph$yy[19], cex = 1.8, pch = 21, col = "black", bg = "red")
points(Phylomorph$xx[20], Phylomorph$yy[20], cex = 1.8, pch = 21, col = "black", bg = "blue")
points(Phylomorph$xx[21:28], Phylomorph$yy[21:28], cex = 1.8, pch = 21, col = "black", bg = "red")
points(Phylomorph$xx[29], Phylomorph$yy[29], cex = 1.8, pch = 21, col = "black", bg = "blue")
points(Phylomorph$xx[30:33], Phylomorph$yy[30:33], cex = 1.8, pch = 21, col = "black", bg = "red")
points(Phylomorph$xx[34:46], Phylomorph$yy[34:46], cex = 1.8, pch = 15, col = "black")
points(Phylomorph$xx[47:48], Phylomorph$yy[47:48], cex = 1.8, pch = 21, col = "black", bg = "green")
points(Phylomorph$xx[49], Phylomorph$yy[49], cex = 1.8, pch = 22, col = "black", bg = "green")
points(Phylomorph$xx[50], Phylomorph$yy[50], cex = 1.8, pch = 21, col = "black", bg = "green")
points(Phylomorph$xx[51:66], Phylomorph$yy[51:66], cex = 2, pch = 21, col = "black", bg = "yellow")
points(Phylomorph$xx[86], Phylomorph$yy[86], cex = 2, pch = 21, col = "black", bg = "yellow")
points(Phylomorph$xx[94], Phylomorph$yy[94], cex = 2, pch = 21, col = "black", bg = "yellow")

#Femur versus MT
Phylomorph <- phylomorphospace(tree,Data[16:17],
                               colors=cols,
                               bty="o",
                               node.by.map=TRUE,
                               node.size = c(1, 0.8),
                               axes = FALSE,
                               ylim = rev(range(Data[,17])),
                               #xlim = c(0, 160),
                               xlab ="", 
                               ylab ="", #Normalised MT Length
                               label = "off")

axis(3, at=seq(30,90,20))
#mtext("Normalised Femur Length", side = 3, line = 3)
axis(2, at=seq(0,200,30))

Plot_ConvexHull(xcoord=Data[FLD_list,][,16],
                ycoord=Data[FLD_list,][,17], bgcolor="blue")
Plot_ConvexHull(xcoord=Data[HLD_list,][,16],
                ycoord=Data[HLD_list,][,17], bgcolor="red")

#Plotting order: Minor Internal nodes -> Taxon Values -> Major Ancestral Nodes
points(Phylomorph$xx[67:85], Phylomorph$yy[67:85], cex = 1.3, pch = 21, col = "black", bg = "blue")
points(Phylomorph$xx[87:93], Phylomorph$yy[87:93], cex = 1.3, pch = 21, col = "black", bg = "red")
points(Phylomorph$xx[95:96], Phylomorph$yy[95:96], cex = 1.3, pch = 21, col = "black", bg = "red")
points(Phylomorph$xx[97], Phylomorph$yy[97], cex = 1.3, pch = 21, col = "black", bg = "black")
points(Phylomorph$xx[98:99], Phylomorph$yy[98:99], cex = 1.3, pch = 21, col = "black", bg = "green")
points(Phylomorph$xx[1:18], Phylomorph$yy[1:18], cex = 1.5, pch = 21, col = "black", bg = "blue")
points(Phylomorph$xx[19], Phylomorph$yy[19], cex = 1.5, pch = 21, col = "black", bg = "red")
points(Phylomorph$xx[20], Phylomorph$yy[20], cex = 1.5, pch = 21, col = "black", bg = "blue")
points(Phylomorph$xx[21:28], Phylomorph$yy[21:28], cex = 1.5, pch = 21, col = "black", bg = "red")
points(Phylomorph$xx[29], Phylomorph$yy[29], cex = 1.5, pch = 21, col = "black", bg = "blue")
points(Phylomorph$xx[30:33], Phylomorph$yy[30:33], cex = 1.5, pch = 21, col = "black", bg = "red")
points(Phylomorph$xx[34:46], Phylomorph$yy[34:46], cex = 1.5, pch = 15, col = "black")
points(Phylomorph$xx[47:48], Phylomorph$yy[47:48], cex = 1.5, pch = 21, col = "black", bg = "green")
points(Phylomorph$xx[49], Phylomorph$yy[49], cex = 1.5, pch = 22, col = "black", bg = "green")
points(Phylomorph$xx[50], Phylomorph$yy[50], cex = 1.5, pch = 21, col = "black", bg = "green")
points(Phylomorph$xx[51:66], Phylomorph$yy[51:66], cex = 1.6, pch = 21, col = "black", bg = "yellow")
points(Phylomorph$xx[86], Phylomorph$yy[86], cex = 1.6, pch = 21, col = "black", bg = "yellow")
points(Phylomorph$xx[94], Phylomorph$yy[94], cex = 1.6, pch = 21, col = "black", bg = "yellow")

#Femur versus MT zoomed
Phylomorph <- phylomorphospace(tree,Data[16:17],
                               colors=cols,
                               bty="o",
                               node.by.map=TRUE,
                               node.size = c(1, 0.8),
                               axes = FALSE,
                               ylim = c(80, 15),
                               xlim = c(55, 85),
                               xlab ="", 
                               ylab ="", #Normalised MT Length
                               label = "off")

axis(3, at=seq(55,85,10))
#mtext("Normalised Femur Length", side = 3, line = 3)
axis(2)

#See Plot B for comments about manually plotting convex hulls
xpoint = c(54.64384, 53, 53, 75.34037, 77.51936, 54.64384)
ypoint = c(28.31892, 28.06146, 83, 83, 72.28355, 28.31892)
polygon(x = xpoint, y = ypoint, col = adjustcolor("blue", alpha.f = 0.2) , border = NA)

xpoint = c(68.68284, 69.01356, 66.66399, 60.07044, 53.85612, 67.79896, 68.68284) 
ypoint = c(71.63045, 61.25585, 54.78554, 50.02363, 83, 83, 71.63045) 
polygon(x = xpoint, y = ypoint, col = adjustcolor("red", alpha.f = 0.2) , border = NA)

#Confidence Intervals
Data <- Data[match(tree$tip.label,rownames(Data)),]
CI.Data <- fastAnc(tree, Data[,16], CI = T )

N14.X <- Phylomorph$xx[64]
N14.Y <- Phylomorph$yy[64]
N15.X <- Phylomorph$xx[65]
N15.Y <- Phylomorph$yy[65]

CI.LO.14 <- CI.Data[["CI95"]][14,1]
CI.UP.14 <- CI.Data[["CI95"]][14,2]
CI.LO.15 <- CI.Data[["CI95"]][15,1]
CI.UP.15 <- CI.Data[["CI95"]][15,2]

plotCI(N14.X,N14.Y, ui = CI.UP.14, li = CI.LO.14, err = "x", lwd = 2.5, add = TRUE)
plotCI(N15.X,N15.Y, ui = CI.UP.15, li = CI.LO.15, err = "x", lwd = 2.5, add = TRUE)

#Plotting order: Minor Internal nodes -> Taxon Values -> Major Ancestral Nodes
points(Phylomorph$xx[67:85], Phylomorph$yy[67:85], cex = 1.6, pch = 21, col = "black", bg = "blue")
points(Phylomorph$xx[87:93], Phylomorph$yy[87:93], cex = 1.6, pch = 21, col = "black", bg = "red")
points(Phylomorph$xx[95:96], Phylomorph$yy[95:96], cex = 1.6, pch = 21, col = "black", bg = "red")
points(Phylomorph$xx[97], Phylomorph$yy[97], cex = 1.6, pch = 21, col = "black", bg = "black")
points(Phylomorph$xx[98:99], Phylomorph$yy[98:99], cex = 1.6, pch = 21, col = "black", bg = "green")
points(Phylomorph$xx[1:18], Phylomorph$yy[1:18], cex = 1.8, pch = 21, col = "black", bg = "blue")
points(Phylomorph$xx[19], Phylomorph$yy[19], cex = 1.8, pch = 21, col = "black", bg = "red")
points(Phylomorph$xx[20], Phylomorph$yy[20], cex = 1.8, pch = 21, col = "black", bg = "blue")
points(Phylomorph$xx[21:28], Phylomorph$yy[21:28], cex = 1.8, pch = 21, col = "black", bg = "red")
points(Phylomorph$xx[29], Phylomorph$yy[29], cex = 1.8, pch = 21, col = "black", bg = "blue")
points(Phylomorph$xx[30:33], Phylomorph$yy[30:33], cex = 1.8, pch = 21, col = "black", bg = "red")
points(Phylomorph$xx[34:46], Phylomorph$yy[34:46], cex = 1.8, pch = 15, col = "black")
points(Phylomorph$xx[47:48], Phylomorph$yy[47:48], cex = 1.8, pch = 21, col = "black", bg = "green")
points(Phylomorph$xx[49], Phylomorph$yy[49], cex = 1.8, pch = 22, col = "black", bg = "green")
points(Phylomorph$xx[50], Phylomorph$yy[50], cex = 1.8, pch = 21, col = "black", bg = "green")
points(Phylomorph$xx[51:66], Phylomorph$yy[51:66], cex = 2, pch = 21, col = "black", bg = "yellow")
points(Phylomorph$xx[86], Phylomorph$yy[86], cex = 2, pch = 21, col = "black", bg = "yellow")
points(Phylomorph$xx[94], Phylomorph$yy[94], cex = 2, pch = 21, col = "black", bg = "yellow")

dev.off()

###Export Phylomorphospace plot of CC_CoM versus Femoral Length (Figure 3E)

tiff(filename = "CoM_Femur_Phylomorphospace.tif", compression = "lzw", width = 222, height = 166, units = "mm", res = 600) #166x166*2 or 166x166*1.5

plot.new()

#par(mai=c(0.5,0.5,0.5,0.5))

Phylomorph <- phylomorphospace(tree,Data[,c(4,16)],
                               colors=cols,
                               bty="o",
                               node.by.map=TRUE,
                               node.size = c(1, 1.2),
                               axes = FALSE,
                               ylim = c(90, 30),
                               #xlim = c(0, 160),
                               xlab ="", 
                               ylab ="Normalised Femur Length",
                               label = "off")

axis(3, at=seq(0,150,50))
mtext("Craniocaudal CoM", side = 3, line = 3)
axis(2, at=seq(30,90,20))

Plot_ConvexHull(xcoord=Data[FLD_list,][,4],
                ycoord=Data[FLD_list,][,16], bgcolor="blue")
Plot_ConvexHull(xcoord=Data[HLD_list,][,4],
                ycoord=Data[HLD_list,][,16], bgcolor="red")

points(Phylomorph$xx[34:46], Phylomorph$yy[34:46], cex = 1.4, pch = 15, col = "black")
points(Phylomorph$xx[49], Phylomorph$yy[49], cex = 1.4, pch = 22, col = "black", bg = "green")
points(Phylomorph$xx[51:66], Phylomorph$yy[51:66], cex = 1.5, pch = 21, col = "black", bg = "yellow")
points(Phylomorph$xx[86], Phylomorph$yy[86], cex = 1.5, pch = 21, col = "black", bg = "yellow")
points(Phylomorph$xx[94], Phylomorph$yy[94], cex = 1.5, pch = 21, col = "black", bg = "yellow")

dev.off()
