%Generate the MMD of branched macroradical products of addition of a
%distribution of macroradicals of one topology to a distribution of chains
%of (the same or different) topology.
%
%SYNTAX
%OUT=radicalgrowth(IN1, OPIN1, OPIN2)
%   IN1 = MMD of one of the reactants:
%       either an nx2 vector of [chain DP, chain fraction] for linear chains or
%       a table for branched chains with the row name reflecting the chain topology (see examples)
%       the radical is assumed to be located at the rightmost arm (terminal macroradicals) 
%           or rightmost branching point (at-branching-point macroradicals)
%   OPIN1 = MMD of the other reactant:
%       can be omitted if IN1 is linear and IN1 == OPIN1 (line 3 in examples.mat)
%   OPIN2 = binding mode (1st letter always refers to MMD in IN1, 2nd letter to MMD in OPIN1)
%       e+e: end to end addition, either IN1 or OPIN1 can define the macroradical
%       e+r: addition of terminal macroradical (IN1) to any location of a chain (OPIN1)
%       x+r: addition of a macroradical with the unpaired electron at the
%           rightmost branching point (IN1) to any location of a chain (OPIN1)
%       r+r: addition of an internal radical to any location of a chain;
%           either IN1 or OPIN1 can define the macroradical
%       x+x: addition of a macroradical with the unpaired electron at the
%           rightmost branching point (IN1) to branch point of a branched chain (OPIN1)
%       if OPIN1 is empty, e+e can be omitted (see examples)
%
%   OUT = table of the product distribution grouped by chain topology
%       the radical is always at the rightmost branch
%
% The code will force size vector of the linear reactant and all products to the uint16 format unless the 2nd reactant uses uint8 for size vectors.
%
% Input MMDs should not have chains with concentrations <1e-12. Product chains with concentration <1e-12 are eliminated
%
% To make the code portable, we removed system memory checks, which may make the code terminate with the "Out of memory" error. The simplest
% work around is to rerun code on a system with more memory; see the "Codefor generating product distributions" of the SI for other strategy to
% reduce memory requirement. All results in the examples.mat file were generated on a PC with 32 Gb of RAM. 

function out=radicalgrowth(mmd1, varargin)
warning('off'), cutoff=1e-12;
%eliminate unsupported combinations
if (nargin==1 && ~isnumeric(mmd1)) || (nargin==2 && any([~isnumeric(mmd1), ~isnumeric(varargin{1})]))
    error('binding pattern is required as a 3rd input if at least one of the input polymers is branched')
end

out=table;
if nargin==1 || (isempty(varargin{1}) && ~contains(varargin{2}, 'r+r')),
    [out.mass{1}, ~, uind]=unique(sort([repelem(mmd1(:,1), size(mmd1,1),1), repmat(mmd1(:,1), size(mmd1,1),1)],2), 'rows');
    out.frac{1}=accumarray(uind, repelem(mmd1(:,2), size(mmd1,1),1).*repmat(mmd1(:,2), size(mmd1,1),1))/sum(mmd1(:,2));
    out.Properties.RowNames{1}='p2x';
elseif nargin==2, mmd2=varargin{1};
    [out.mass{1}, ~, uind]=unique(sort([repelem(mmd1(:,1), size(mmd2,1),1), repmat(mmd2(:,1), size(mmd1,1),1)],2),'rows');
    out.frac{1}=accumarray(uind, repelem(mmd1(:,2), size(mmd2,1),1).*repmat(mmd2(:,2), size(mmd1,1),1))/max([sum(mmd1(:,2)), sum(mmd2(:,2))]);
    out.Properties.RowNames{1}='p2x';
elseif contains(varargin{2}, 'r+r'), out=table; [pxp1, tfrac1]=getpxp(mmd1, 'linear');
    if ~isempty(varargin{1}), mmd2=varargin{1}; [pxp2, tfrac2]=getpxp(mmd2, 'linear');
    else pxp2=pxp1; tfrac2=tfrac1; mmd2=mmd1; end
    [out.mass{1}, ~, uind]=unique(sort([repelem(pxp1, size(pxp2,1),1), repmat(pxp2, size(pxp1,1),1)],2), 'rows');
    out.frac{1}=accumarray(uind, repelem(tfrac1, size(pxp2,1),1).*repmat(tfrac2, size(pxp1,1),1));
    [out.mass{2}, ~, uind]=unique(sort([repelem(uint16(mmd1(:,1)), size(pxp2,1),1), repmat(pxp2, size(mmd1(:,1),1),1)],2), 'rows');
    out.frac{2}=accumarray(uind, repelem(2*mmd1(:,2)./mmd1(:,1), size(pxp2,1),1).*repmat(tfrac2, size(mmd1,1),1));
    clear pxp2 tfrac2
    [out.mass{2}, ~, uind]=unique([out.mass{2}; sort([repelem(uint16(mmd2(:,1)), size(pxp1,1),1), repmat(pxp1, size(mmd2(:,1),1),1)],2)], 'rows');
    out.frac{2}=accumarray(uind, [out.frac{2}; repelem(2*mmd2(:,2)./mmd2(:,1), size(pxp1,1),1).*repmat(tfrac1, size(mmd2,1),1)]);
    clear pxp1 tfrac1
    [out.mass{3}, ~, uind]=unique(sort([repelem(uint16(mmd1(:,1)), size(mmd2,1),1), repmat(uint16(mmd2(:,1)), size(mmd1,1),1)],2), 'rows');
    out.frac{3}=accumarray(uind, repelem(2*mmd1(:,2)./mmd1(:,1), size(mmd2,1),1).*repmat(2*mmd2(:,2)./mmd2(:,1), size(mmd1,1),1));
    out.Properties.RowNames={'p4x', 'p3x', 'p2x'};
    out.frac=cellfun(@times, out.frac, repmat({1./max([sum(mmd1(:,2)), sum(mmd2(:,2))])},3,1), 'UniformOutput', 0);
elseif contains(varargin{2}, 'e+e'),
    mmd2=varargin{1}; out=table;
    if isnumeric(mmd2), tmmd=mmd1; mmd1=mmd2; mmd2=tmmd; clear tmmd, end
    if isnumeric(mmd1)
        out.mass{1}=repelem(mmd2.mass{1}, size(mmd1,1),1); out.mass{1}(:,end)=out.mass{1}(:,end)+repmat(uint16(mmd1(:,1)), size(mmd2.mass{1}, 1),1);
        out.frac{1}=repelem(mmd2.frac{1}, size(mmd1,1),1).*repmat(mmd1(:,2), size(mmd2.frac{1}));
        out.frac{1}=out.frac{1}/max([sum(mmd2.frac{1}), sum(mmd1(:,2))]);
        out.Properties.RowNames=mmd2.Properties.RowNames;
        %test
        if size(out.mass{1},2)~=size(mmd2.mass{1},2), error('unexpected number of columns for le+be addition'), end
    else,
        [p1,x1]=parsename(mmd1.Properties.RowNames{1}); [p2,x2]=parsename(mmd2.Properties.RowNames{1});
        if length(p2)<length(p1), tmmd=mmd2; mmd2=mmd1; mmd1=tmmd; tmmd={p2, x2}; p2=p1; x2=x1; p1=tmmd{1}; x1=tmmd{2}; end
        out.mass{1}=repelem(mmd2.mass{1}, size(mmd1.mass{1},1),1); out.mass{1}(:,end)=out.mass{1}(:,end)+repmat(mmd1.mass{1}(:,end), size(mmd2.mass{1}, 1),1);
        if length(p1)==1, mmd1.mass{1}=mmd1.mass{1}(:, 1:end-1); if length(p2)==1, name={[p2-1, 1, p1-1]}; else name={[p2, p1-1]}; end, name{2}=[x2, x1];
        elseif length(p1)==3, mmd1.mass{1}=mmd1.mass{1}(:, [end-p1(end)+1:end-1, p1(1)+1, 1:p1(1)]); name={[p2, p1(3), p1(1)], [x2, flip(x1)]};
        elseif length(p1)==4, mmd1.mass{1}=mmd1.mass{1}(:, [end-p1(end)+1:end-1, p1(1)+2:p1(1)+p1(3), p1(1)+1, 1:p1(1)]);
            name={[p2, p1([4,3]), p1(1)], [x2, flip(x1)]};
        else mmd1.mass{1}=mmd1.mass{1}(:, 1:end-1); name={[p2, p1([1,3:end])], [x2, x1]};
        end
        out.mass{1}=[out.mass{1}, repmat(mmd1.mass{1}, size(mmd2.mass{1}, 1),1)];
        out.frac{1}=repelem(mmd2.frac{1}, size(mmd1.frac{1},1),1).*repmat(mmd1.frac{1}, size(mmd2.frac{1}));
        out.frac{1}=out.frac{1}/max([sum(mmd2.frac{1}), sum(mmd1.frac{1})]);
        out.Properties.RowNames{1}=generatename(name{1}, name{2});
        %test
        if size(out.mass{1},2)~=size(mmd1.mass{1},2)+size(mmd2.mass{1},2), error('unexpected number of columns for be+be addition'), end
    end
    out=cleanbranch(out);
    
elseif contains(varargin{2}, 'x+x'),
    mmd2=varargin{1}; out=table;
    [p1,x1]=parsename(mmd1.Properties.RowNames{1}); [p2,x2]=parsename(mmd2.Properties.RowNames{1});
    if length(p2)<length(p1), tmmd=mmd2; mmd2=mmd1; mmd1=tmmd; tmmd={p2, x2}; p2=p1; x2=x1; p1=tmmd{1}; x1=tmmd{2}; end
    if length(p1)==1, if length(p2)==1, name={p2+p1}; else name={[p2(1:end-1), p2(end)+p1]}; end, name{2}=[x2(1:end-1), x2(end)+x1];
    else, name={[p2(1:end-1), p2(end)+p1(end)+1], [x2(1:end-1), x2(end)+x1(end), flip(x1(1:end-1))]};
        if length(p1)==3, name{1}=[name{1}, p1(1)]; mmd1.mass{1}=mmd1.mass{1}(:, [end-p1(3)+1:end, p1(1)+1, 1:p1(1)]);
        elseif length(p1)==4, name{1}=[name{1}, p1(3), p1(1)];
            mmd1.mass{1}=mmd1.mass{1}(:, [end-p1(end)+1:end, end-p1(end), p1(1)+2:p1(1)+p1(3), p1(1)+1, 1:p1(1)]);
        else name{1}=[name{1}-p1(end)+p1(1), p1(2:end)];
        end
    end
    out.mass{1}=[repelem(mmd2.mass{1}, size(mmd1.mass{1},1),1), repmat(mmd1.mass{1}, size(mmd2.mass{1}, 1),1)];
    out.frac{1}=repelem(mmd2.frac{1}, size(mmd1.frac{1},1),1).*repmat(mmd1.frac{1}, size(mmd2.frac{1}));
    out.frac{1}=out.frac{1}/max([sum(mmd2.frac{1}), sum(mmd1.frac{1})]);
    out.Properties.RowNames{1}=generatename(name{1}, name{2});
    %test
    if size(out.mass{1},2)~=size(mmd1.mass{1},2)+size(mmd2.mass{1},2), error('unexpected number of columns for be+be addition'), end
    out=cleanbranch(out);
    
elseif contains(varargin{2}, {'e+x', 'x+e'}),
    if startsWith(varargin{2}, 'x'), mmd2=mmd1; mmd1=varargin{1}; else, mmd2=varargin{1}; end
    out=table;
    [p2,x2]=parsename(mmd2.Properties.RowNames{1});
    if isnumeric(mmd1), name=generatename([p2(1:end-1), p2(end)+1], x2); mmd1=table({uint16(mmd1(:,1))}, {mmd1(:,2)}, 'VariableNames', {'mass', 'frac'});
    else, [p1,x1]=parsename(mmd1.Properties.RowNames{1});
        if length(p1)==1, if length(p2)==1, name=generatename([p2, 1, p1-1], [x2,x1]); else, name=generatename([p2(1:end-1), p2(end)+1, p1-1], [x2,x1]); end
            mmd1.mass{1}=mmd1.mass{1}(:, [end, 1:end-1]);
        else name=generatename([p2(1:end-1), p2(end)+1, p1(end:-1:3), p1(1)], [x2(1:end-1), x2(end), flip(x1)]);
            if length(p1)==3, mmd1.mass{1}=mmd1.mass{1}(:, [end-p1(3)+1:end, p1(1)+1, 1:p1(1)]);
            elseif length(p1)==4, mmd1.mass{1}=mmd1.mass{1}(:, [end-p1(end)+1:end, end-p1(end), p1(1)+2:p1(1)+p1(3), p1(1)+1, 1:p1(1)]);
            end
        end
    end
    out.mass{name}=[repelem(mmd2.mass{1}, size(mmd1.mass{1},1),1), repmat(mmd1.mass{1}, size(mmd2.mass{1},1),1)];
    out.frac{name}=repelem(mmd2.frac{1}, size(mmd1.mass{1},1),1).*repmat(mmd1.frac{1}, size(mmd2.mass{1},1),1)/max([sum(mmd2.frac{1}), sum(mmd1.frac{1})]);
    %test
    if size(out.mass{1},2)~=size(mmd1.mass{1},2)+size(mmd2.mass{1},2), error('unexpected number of columns for be+be addition'), end
    out=cleanbranch(out);
    
else
    if startsWith(varargin{2}, {'e', 'x'}), mmd2=varargin{1}; else mmd2=mmd1; mmd1=varargin{1}; end
    
    if istable(mmd1), [p1,x1]=parsename(mmd1.Properties.RowNames{1});
        if contains(varargin{2}, 'e')
            if length(p1)==1, newp={[p1-1, 1,2], [p1-1,1,1], [p1-1,1]};  else newp={[p1,2], [p1,1], p1}; end
            x1=[x1, 1];
        elseif length(p1)==1, newp={[p1+1,1], [p1,1], p1};
        else, newp={[p1(1:end-1), p1(end)+2], [p1(1:end-1), p1(end)+1], p1};
        end
        
    else newp={[2 1], [1, 1], 1}; mmd1=table({uint16(mmd1(:,1))}, {mmd1(:,2)}, 'VariableNames', {'mass', 'frac'}); x1=1;
    end
    
    if ~istable(mmd2),  [pxp, tfrac]=getpxp(mmd2, 'linear');
        out.mass{1}=[repelem(mmd1.mass{1}, size(pxp,1),1), repmat(pxp, size(mmd1.mass{1},1),1)]; pxp=length(pxp);
        out.frac{1}=repelem(mmd1.frac{1}, pxp,1).*repmat(tfrac, size(mmd1.mass{1},1),1); clear tfrac
        out.mass{2}=[repelem(mmd1.mass{1}, size(mmd2,1),1), repmat(uint16(mmd2(:,1)), size(mmd1.mass{1},1),1)];
        out.frac{2}=repelem(mmd1.frac{1}, size(mmd2,1),1).*repmat(2*mmd2(:,2)./(mmd2(:,1)+1), size(mmd1.mass{1},1),1);
        if size(mmd1.mass{1},2)==1, out.Properties.RowNames={'p3x', 'p2x'};
        elseif length(p1)==1 && contains(varargin{2}, 'x'), out.Properties.RowNames={generatename(newp{1,1}(1)+1, x1), generatename(newp{1,2}(1)+1, x1)}; %b1x+lr
        else, out.Properties.RowNames={generatename(newp{1,1}, x1), generatename(newp{1,2}, x1)}; %if e+lr, 1 additional branch point is created, with x1 already expanded, for x+lr, no new branch points
        end
        
    else, [p2,x2]=parsename(mmd2.Properties.RowNames{1});
        if length(p2)==1
            newp(1:2)={[newp{1}, p2-1], [newp{2}, p2-1]};
            if contains(varargin{2}, 'e') && length(newp{3})>1, newp{3}=[newp{3},p2]; else newp{3}=[newp{3}(1:end-1), newp{3}(end)+p2];  end
            newx={[x1, x2], [x1, x2]}; x1(end)=x1(end)+x2; newx{3}=x1; x1(end)=x1(end)-x2;
        else, tmp={[newp{1}, p2([1,3:end])], [newp{2}, p2([1,3:end])]};
            if length(newp{3})==1, tmp{3}=[p2(1)+newp{3}, p2(2:end)]; elseif contains(varargin{2}, 'x'), tmp{3}=[newp{3}(1:end-1), newp{3}(end)+p2(1)+1, p2(3:end)];
            else tmp{3}=[newp{3}, p2(1)+1, p2(3:end)]; end
            newp{1}=flip(newp{1}([1,3:end])); newp{2}=flip(newp{2}([1,3:end])); if length(newp{3})>1, newp{3}=flip(newp{3}([1,3:end])); end
            tmp(2,1:2)={[p2, newp{1}], [p2, newp{2}]};
            if ~exist('p1', 'var') || (contains(varargin{2}, 'x') && length(newp{3})==1), tmp{2,3}=[p2(1:end-1), p2(end)+newp{3}];
            elseif contains(varargin{2}, 'e'), tmp{2,3}=[p2(1:end-1), p2(end)+1, newp{3}]; else tmp{2,3}=[p2(1:end-1), p2(end)+1+newp{3}(1), newp{3}(2:end)];
            end
            newp=tmp;
            newx=repmat({[x1, x2]; [x2, x1(end:-1:1)]}, 1,2); x1(end)=x1(end)+x2(1); newx{1,3}=[x1, x2(2:end)];
            x1(end)=x1(end)-x2(1)+x2(end); newx{2,3}=[x2(1:end-1), x1(end:-1:1)]; x1(end)=x1(end)-x2(end);
        end
        if size(newp,1)==3, newx(3,1)=newx(1,1); end
        %checks
        tmp=cellfun(@sum, newp);
        if any(tmp(:,1)~=(size(mmd1.mass{1}, 2)+size(mmd2.mass{1}, 2)+1)) || any(any(tmp(:,2:end)~=(size(mmd1.mass{1}, 2)+size(mmd2.mass{1}, 2))))
            error('newp doesnt match the expected number of arms in the product'), end
        tmp=cellfun(@sum, newx);
        if size(mmd1.mass{1},2)==1, x1=[]; else [p1,x1]=parsename(mmd1.Properties.RowNames{1}); end
        if (~contains(varargin{2}, 'x') && any(any(tmp~=(sum(x1)+sum(x2)+1)))) || (contains(varargin{2}, 'x') && any(any(tmp~=(sum(x1)+sum(x2)))))
            error('newx doesnt match the expected number of branch points in the product')
        end
        
        out.mass=cell(3,1); out.frac=cell(3,1);
        
        mmd2.frac{1}=mmd2.frac{1}./(sum(mmd2.mass{1}, 2)+1); %this will result in loss of fraction because portions of branched chains are not considered for binding
        for j=p2(1):-1:1
            [pxp, uamass, uind]=getpxp(mmd2.mass{1}(:,j));
            [out.mass{1}, ~, tuind]=unique([out.mass{1}; [repelem(mmd1.mass{1}, length(pxp),1), repmat([repelem(mmd2.mass{1}(:, j), uamass(uind),1)-pxp, pxp,...
                repelem(mmd2.mass{1}(:, setdiff(1:size(mmd2.mass{1}, 2), j)), uamass(uind),1)], size(mmd1.mass{1},1),1)]], 'rows');
            pxp=length(pxp);
            out.frac{1}=accumarray(tuind, [out.frac{1}; repmat(repelem(mmd2.frac{1}, uamass(uind), 1), size(mmd1.mass{1},1),1).*repelem(mmd1.frac{1}, pxp,1)]);
            clear tuind uind uamass pxp
            [out.mass{2}, ~, tuind]=unique([out.mass{2}; [repelem(mmd1.mass{1}, length(mmd2.mass{1}),1), repmat(mmd2.mass{1}(:, [j, setdiff(1:size(mmd2.mass{1}, 2), j)]),...
                size(mmd1.mass{1},1),1)]], 'rows');
            out.frac{2}=accumarray(tuind, [out.frac{2}; repelem(mmd1.frac{1}, length(mmd2.mass{1}),1).*repmat(mmd2.frac{1}, size(mmd1.mass{1},1),1)]);
            clear tuind
        end
        [out.mass{3}, ~, tuind]=unique([out.mass{3}; [repelem(mmd1.mass{1}, length(mmd2.mass{1}),1), repmat(mmd2.mass{1},size(mmd1.mass{1},1),1)]], 'rows');
        out.frac{3}=accumarray(tuind, [out.frac{3}; repelem(mmd1.frac{1}, length(mmd2.mass{1}),1).*repmat(mmd2.frac{1}, size(mmd1.mass{1},1),1)]);
        
        for j=1:3,
            if length(newp{1,j})==3, out.mass{j}=out.mass{j}(:, [end-newp{1,j}(end)+1:end, newp{1,j}(1)+1, 1:newp{1,j}(1)]); newp{1,j}=newp{1,j}(end:-1:1);
                newx{1,j}=newx{1,j}(end:-1:1);
            elseif length(newp{1,j})==4, out.mass{j}=out.mass{j}(:, [end-newp{1,j}(end)+1:end, end-newp{1,j}(end), newp{1,j}(1)+2:newp{1,j}(1)+newp{1,j}(3),...
                    newp{1,j}(1)+1, 1:newp{1,j}(1)]); newp{1,j}=[newp{1,j}(4), 1, newp{1,j}([3,1])]; newx{1,j}=newx{1,j}(end:-1:1);
            end
        end
        
        namesp=cellfun(@generatename, newp, newx, 'UniformOutput', 0);
        out.Properties.RowNames=namesp(1,:);
        tmp=unique(setdiff(namesp(2:end,:), namesp(1,:)));
        if ~isempty(tmp), out(end+1:end+length(tmp),:)=cell2table(cell(length(tmp),2)); out.Properties.RowNames(4:length(tmp)+3)=tmp; end
        
        if size(newp,1)>1
            if ~exist('p1', 'var')
                newp{3,1}=[p2(1),1,2,p2(3:end)]; newx{3,1}=[x2(1),1,x2(2:end)];
                if length(p2)==4, newp{3,2}=[p2(1:3),2,p2(4)]; newx{3,2}=[x2(1:2),1,x2(3)];
                    newp(4,:)={[p2(1),1,1,2,p2(3)-1,p2(4)], [p2(1),1,1,1,p2(3)-1,p2(4)], [p2(1),1,p2(3)+1,p2(4)]};
                    newx{3,2}=[x2(1:2),1,x2(3)]; newx(4,:)={[x2(1:2), 1, x2(3)], [x2(1:2),1,x2(3)], [x2(1), x2(2)+1, x2(3)]};
                end
                %test
                tmp=cellfun(@sum, newp(3:end,:));
                if (size(tmp,1)==1 && tmp(1)~=size(mmd2.mass{1},2)+2) || (size(tmp,1)>1 && (any([tmp(1,1:2), tmp(2,1)]~=(size(mmd2.mass{1},2)+2)) || any(tmp(2,2:3)~=(1+size(mmd2.mass{1},2)))))
                    error('newp for addition to bridging/central branches of b2/b3 doesnt match the expected number of arms in the product'), end
                tmp=cellfun(@sum, newx(3:end,:));
                if any(setdiff(tmp(:,3:end),0)~=(sum(x2)+1)), error('newx for addition to bridging/central branches of b2/b3 doesnt match the expected number of arms in the product'), end
                namesp(3:size(newp,1),:)=cellfun(@generatename, newp(3:end,:), newx(3:end,:), 'UniformOutput', 0);
                tmp=unique(setdiff(namesp(3:end,:), namesp(1:2,:))); tmp(cellfun('isempty', tmp))=[];
                if ~isempty(tmp), tmp=[out.Properties.RowNames; tmp]; out(end+1:end+length(tmp)-height(out),:)=cell2table(cell(length(tmp)-height(out),2));
                    out.Properties.RowNames=tmp; end
                clear tmp
                
                [pxp, uamass, uind]=getpxp(mmd2.mass{1}(:,p2(1)+1)); %xpx is the length of p
                [out.mass{namesp{3,1}}, ~, tuind]=unique([out.mass{namesp{3,1}}; [repmat([repelem(mmd2.mass{1}(:, 1:p2(1)), uamass(uind),1), repelem(mmd2.mass{1}(:, p2(1)+1),...
                    uamass(uind),1)-pxp], size(mmd1.mass{1},1),1), repelem(mmd1.mass{1}, length(pxp),1),...
                    repmat([pxp,repelem(mmd2.mass{1}(:, p2(1)+2:end), uamass(uind),1)], size(mmd1.mass{1},1),1)]], 'rows');
                out.frac{namesp{3,1}}=accumarray(tuind, [out.frac{namesp{3,1}}; repmat(repelem(mmd2.frac{1}, uamass(uind), 1), size(mmd1.mass{1},1),1).*...
                    repelem(mmd1.frac{1}, length(pxp),1)]);
                clear tuind uamass uind pxp
                
                if length(p2)==4
                    [pxp, uamass, uind]=getpxp(mmd2.mass{1}(:,p2(1)+p2(3)+1));
                    [out.mass{namesp{3,2}}, ~, tuind]=unique([out.mass{namesp{3,2}}; [repmat([repelem(mmd2.mass{1}(:, 1:p2(1)+p2(3)), uamass(uind),1), repelem(mmd2.mass{1}(:, p2(1)+p2(3)+1),...
                        uamass(uind),1)-pxp], size(mmd1.mass{1},1),1), repelem(mmd1.mass{1}, length(pxp),1),repmat([pxp,repelem(mmd2.mass{1}(:, p2(1)+p2(3)+2:end), uamass(uind),1)],...
                        size(mmd1.mass{1},1),1)]], 'rows');
                    out.frac{namesp{3,2}}=accumarray(tuind, [out.frac{namesp{3,2}}; repmat(repelem(mmd2.frac{1}, uamass(uind), 1), size(mmd1.mass{1},1),1).*repelem(mmd1.frac{1}, length(pxp),1)]);
                    for j=p2(1)+2:p2(1)+p2(3)
                        [pxp, uamass, uind]=getpxp(mmd2.mass{1}(:,j));
                        [out.mass{namesp{4,1}}, ~, tuind]=unique([repmat([repelem(mmd2.mass{1}(:, 1:p2(1)+1), uamass(uind),1), repelem(mmd2.mass{1}(:, j),...
                            uamass(uind),1)-pxp], size(mmd1.mass{1},1),1), sort([repmat(pxp,size(mmd1.mass{1},1),1), repelem(mmd1.mass{1}, length(pxp),1)],2),...
                            repmat(repelem(mmd2.mass{1}(:, setdiff(p2(1)+2:sum(p2), j)), uamass(uind),1), size(mmd1.mass{1},1),1)], 'rows');
                        out.frac{namesp{4,1}}=accumarray(tuind, repmat(repelem(mmd2.frac{1}, uamass(uind), 1), size(mmd1.mass{1},1),1).*repelem(mmd1.frac{1}, length(pxp),1));
                        clear tuind uind uamass pxp
                        [out.mass{namesp{4,2}}, ~, tuind]=unique([repmat(mmd2.mass{1}(:, [1:p2(1)+1,j]), size(mmd1.mass{1},1),1),...
                            repelem(mmd1.mass{1}, size(mmd2.mass{1},1),1), repmat(mmd2.mass{1}(:, setdiff(p2(1)+2:sum(p2), j)), size(mmd1.mass{1},1),1)], 'rows');
                        out.frac{namesp{4,2}}=accumarray(tuind, repmat(mmd2.frac{1}, size(mmd1.mass{1},1),1).*repelem(mmd1.frac{1}, size(mmd2.mass{1},1),1));
                        clear tuind
                    end
                    [out.mass{namesp{4,3}}, ~, tuind]=unique([repmat(mmd2.mass{1}(:,1:p2(1)+1),size(mmd1.mass{1},1),1), repelem(mmd1.mass{1}, length(mmd2.mass{1}),1),...
                        repmat(mmd2.mass{1}(:,p2(1)+2:end),size(mmd1.mass{1},1),1)], 'rows');
                    out.frac{namesp{4,3}}=accumarray(tuind, repelem(mmd1.frac{1}, length(mmd2.mass{1}),1).*repmat(mmd2.frac{1}, size(mmd1.mass{1},1),1));
                    clear tuind
                    out(namesp{4,3},:)=cleanbranch(out(namesp{4,3},:));
                end
            else
                if length(p1)<3, mmd1.mass{1}=mmd1.mass{1}(:, [end, 1:end-1]);
                elseif length(p1)==3, mmd1.mass{1}=mmd1.mass{1}(:, [end, end-p1(end)+1:end-1, p1(1)+1, 1:p1(1)]);
                elseif length(p1)==4, mmd1.mass{1}=mmd1.mass{1}(:, [end, end-p1(end)+1:end-1, end-p1(end), p1(1)+2:p1(1)+p1(3), p1(1)+1, 1:p1(1)]);
                end
            end
            
            for j=sum(p2)-p2(1)+1:sum(p2)
                [uamass, ~, uind]=unique(mmd2.mass{1}(:,j));
                pxp=cellfun(@transpose, cellfun(@colon, repmat({1}, size(uamass)), num2cell(uamass-1), 'UniformOutput', 0), 'UniformOutput', 0);
                pxp=cell2mat(pxp(uind));
                uamass=uamass-1;
                [out.mass{namesp{2,1}}, ~, tuind]=unique([out.mass{namesp{2,1}}; [repmat([repelem(mmd2.mass{1}(:, setdiff(1:sum(p2), j)),uamass(uind),1), ...
                    repelem(mmd2.mass{1}(:, j), uamass(uind),1)-pxp, pxp], size(mmd1.mass{1},1),1), repelem(mmd1.mass{1}, length(pxp),1)]], 'rows');
                pxp=length(pxp);
                out.frac{namesp{2,1}}=accumarray(tuind, [out.frac{namesp{2,1}}; repmat(repelem(mmd2.frac{1}, uamass(uind), 1), size(mmd1.mass{1},1),1).*...
                    repelem(mmd1.frac{1}, pxp,1)]);
                [out.mass{namesp{2,2}}, ~, tuind]=unique([out.mass{namesp{2,2}}; [repmat(mmd2.mass{1}(:, [setdiff(1:sum(p2), j),j]),size(mmd1.mass{1},1),1),...
                    repelem(mmd1.mass{1}, length(mmd2.mass{1}),1)]], 'rows');
                out.frac{namesp{2,2}}=accumarray(tuind, [out.frac{namesp{2,2}}; repmat(mmd2.frac{1}, size(mmd1.mass{1},1),1).*repelem(mmd1.frac{1},...
                    length(mmd2.mass{1}),1)]);
                clear tuind
            end
            [out.mass{namesp{2,3}}, ~, tuind]=unique([out.mass{namesp{2,3}}; [repmat(mmd2.mass{1},size(mmd1.mass{1},1),1), repelem(mmd1.mass{1}, length(mmd2.mass{1}),1)]], 'rows');
            out.frac{namesp{2,3}}=accumarray(tuind, [out.frac{namesp{2,3}}; repelem(mmd1.frac{1}, length(mmd2.mass{1}),1).*repmat(mmd2.frac{1}, size(mmd1.mass{1},1),1)]);
            clear tuind
        end
    end
    for j=height(out):-1:1, tmp=cleanbranch1(out(j,:), 'noflip'); out=[out(setdiff(1:height(out), j),:); tmp]; end
end

%final tests and cleanup
for i=1:height(out), ind=out.frac{i}<=cutoff; tmass=sum(out.mass{i}(ind,:),2)'*out.frac{i}(ind); out.mass{i}(ind,:)=[]; out.frac{i}(ind)=[];
    tmass(2)=sum(out.mass{i},2)'*out.frac{i};
    out.frac{i}=out.frac{i}/tmass(2)*sum(tmass);
end
if istable(mmd1) && istable(mmd2) && ~contains(varargin{2}, {'e+x', 'x+e', 'x+x'}), out.frac=cellfun(@times, out.frac, repmat({1./max([sum(mmd1.frac{1}), sum(mmd2.frac{1})])}, height(out), 1), 'UniformOutput', 0); end

ind=cellfun('isempty', out.mass);
if any(ind)
    if cellfun('size', regexp(out.Properties.RowNames(ind), 'p'), 2)<5, error('some of the expected product chain topology are missing')
    else out(ind,:)=[]; end, end
clear tmp
[tmp(:,1), tmp(:,2)]=cellfun(@parsename, out.Properties.RowNames, 'UniformOutput', 0);
tmp=cellfun(@sum, tmp);
if length(unique(diff(tmp(:,2))))>1, error('inconsistent set of branch point indeces')
elseif any(cellfun('size', out.mass, 2)~=tmp(:,1)), error('chain names are inconsistent with the number of columns defining them')
end



    function [p,x]=parsename(name)
        p=str2double(regexprep(regexp(name, 'p[0-9]{0,}', 'match'), 'p', '')); p(isnan(p))=1;
        x=str2double(regexprep(regexp(name, 'x[0-9]{0,}', 'match'), 'x', '')); x(isnan(x))=1;
    end

    function name=generatename(p, x)
        if length(p)-length(x)==2 %branched side chain
            tmp=[repmat({'p'}, length(x)+1, 1), cellstr(num2str(p(1:length(x)+1)'))]; tmp([1:3,end], 3:4)=[repmat({'x'}, length(x), 1), cellstr(num2str(x'))];
            tmp(end-1, 3:4)={''}; name=char(join(join(tmp,''),''));
        else, name=char(join(join([repmat({'p'}, length(x), 1), cellstr(num2str(p(1:length(x))')), repmat({'x'}, length(x), 1), cellstr(num2str(x'))], ''), ''));
        end
        if length(p)>length(x) name=[name, 'p', num2str(p(end))]; end
        name=regexprep(regexprep(name, ' ', ''), '(?<=[px])1(?=[px]|$)', '');
    end

    function [pxp, uamass, uind]=getpxp(mmd, varargin)
        if nargin>1 && contains(varargin{1}, 'linear'), uind=[];
            pxp=cell2mat(cellfun(@transpose, cellfun(@colon, repmat({1}, size(mmd,1),1), num2cell(uint16(floor(mmd(:,1)/2))), 'UniformOutput', 0), 'UniformOutput', 0));
            pxp(:,2)=repelem(uint16(mmd(:,1)), floor(mmd(:,1)/2))-pxp;
            uamass=repelem(mmd(:,2)./(mmd(:,1)+1), floor(mmd(:,1)/2), 1); ind=pxp(:,1)==pxp(:,2); uamass(~ind)=uamass(~ind)*2;
        else
            [uamass, ~, uind]=unique(mmd);
            pxp=cellfun(@transpose, cellfun(@colon, repmat({1}, size(uamass)), num2cell(uamass-1), 'UniformOutput', 0), 'UniformOutput', 0);
            pxp=cell2mat(pxp(uind));
            uamass=uamass-1;
        end
    end

    function  [mass, p, x]=removedangling(mass, p, x)
        if all(p==1), x=0;
        else
            if p(1)==1, mass(:,2)=mass(:,2)+mass(:,1); mass(:,1)=[];
                if length(p)==3, p=p(3)+1; x=sum(x);  else p=p([3,2,4:end]); x(2)=x(1)+x(2); x(1)=[];  end, end
            if p(end)==1 && length(p)>2, tind=length(p)+1-find(cumsum(flip(p))>(1:length(p)), 1, 'first'); p(tind)=p(tind)+1; p(tind+1:end)=[];
                if tind==1, x=sum(x); else x(tind-1)=sum(x(tind-1:end)); x(tind:end)=[]; end
                tind=sum(p(1:tind)); mass(:, tind)=sum(mass(:, tind:end), 2); mass(:,tind+1:end)=[];  end
            if length(p)==2, p=p(1)+p(2); x=sum(x); end
        end
    end

    function mmd=cleanbranch1(mmd, varargin)
        name=mmd.Properties.RowNames{1};
        if isempty(regexp(name, 'p[0-9]?p', 'once')),
            p=str2double(regexprep(regexp(name, 'p[0-9]{0,}', 'match'), 'p', '')); p(isnan(p))=1;
            x=str2double(regexprep(regexp(name, 'x[0-9]{0,}', 'match'), 'x', '')); x(isnan(x))=1;
            if endsWith(name, 'x') && length(p)>1, name(end)=[]; mmd.Properties.RowNames{1}(end)=[]; x(end)=[]; end
            if nargin==1 || ~contains(varargin{1}, 'keepdangling')
                [mmd.mass{1}, p, x]=removedangling(mmd.mass{1}, p, x);
                mmd.Properties.RowNames{1}=generatename(p, x);
            end
            mmd.p{1}=p; mmd.x{1}=x;
            if length(p)>1 && p(2)~=1, save('cleanbrancherror.mat'), error('cleanbranch: incorrect polymer name'), end
            
            ind=mmd.mass{1}(:,end)==0;
            if any(ind), mmd.mass{2}=mmd.mass{1}(ind, 1:end-1); mmd.mass{1}(ind, :)=[]; mmd.frac{2}=mmd.frac{1}(ind); mmd.frac{1}(ind)=[]; clear ind
                mmd.p{2}=p(1:end-1); if p(end)>1, mmd.p{2}(end+1)=p(end)-1; end,
                if nargin==1 || ~contains(varargin{1}, 'keepdangling'), [mmd.mass{2}, mmd.p{2}, mmd.x{2}]=removedangling(mmd.mass{2}, mmd.p{2}, x); end
            end
            if any(length(p)==[4,5]), ind=mmd.mass{1}(:,p(1)+p(3))==0;
                if any(ind), mmd.p{end+1}=[p(1:2), p(3)-1, p(4:end)];
                    mmd.mass{end}=mmd.mass{1}(ind, [1:p(1)+p(3)-1, p(1)+p(3)+1:end]); mmd.mass{1}(ind, :)=[]; mmd.frac{end}=mmd.frac{1}(ind); mmd.frac{1}(ind)=[];
                end
                if length(p)==5, ind=mmd.mass{1}(:,p(1)+p(3)+p(4))==0;
                    if any(ind), mmd.p{end+1}=[p(1:3), p(4)-1, p(5)];
                        mmd.mass{end}=mmd.mass{1}(ind, [1:p(1)+p(3)+p(4)-1, p(1)+p(3)+p(4)+1:end]); mmd.mass{1}(ind, :)=[]; mmd.frac{end}=mmd.frac{1}(ind); mmd.frac{1}(ind)=[];
                    end
                end
            end
            mmd.x(cellfun('isempty', mmd.x))=mmd.x(1);
            for i=1:height(mmd)
                p=mmd.p{i}; x=mmd.x{i};
                if length(p)<3, mmd.mass{i}(:,1:p(1))=sort(mmd.mass{i}(:,1:p(1)),2);
                else mmd.mass{i}(:, 1:p(1))=sort(mmd.mass{i}(:, 1:p(1)),2); mmd.mass{i}(:, end-p(end)+1:end)=sort(mmd.mass{i}(:, end-p(end)+1:end),2); %sort branches
                    if length(p)>=4 && p(3)>2, mmd.mass{i}(:, p(1)+2:p(1)+p(3)+1)=sort(mmd.mass{i}(:, p(1)+2:p(1)+p(3)+1),2); end
                    if length(p)>=5 && p(4)>2, mmd.mass{i}(:, p(1)+p(3)+2:sum(p(1:4)))=sort(mmd.mass{i}(:, p(1)+p(3)+2:sum(p(1:4))),2); end
                end
                
                if nargin==1 || ~contains(varargin{1}, 'noflip')
                    ind=false(size(mmd.mass{i},1),1);
                    if length(p)==3
                        if p(1)>p(3) || (p(1)==p(3) && x(1)>x(2)), ind=true(size(mmd.mass{i},1),1); p([1,3])=p([3,1]); x=x([2,1]);
                        elseif p(1)==p(3) && x(1)==x(2), ind=mmd.mass{i}(:,p(1))>mmd.mass{i}(:,end); end
                        mmd.mass{i}(ind,:)=mmd.mass{i}(ind, [end-p(3)+1:end, p(1)+1, 1:p(1)]);
                    elseif length(p)==4
                        if p(3)>2, mmd.mass{i}(:, p(1)+2:p(1)+p(3))=sort(mmd.mass{i}(:, p(1)+2:p(1)+p(3)), 2); end
                        if p(1)>p(end)|| (p(1)==p(end) && x(1)>x(end)), ind=true(size(mmd.mass{i},1),1);  p=[p(end), 1, p([3,1])]; x([1,3])=x([3,1]);
                        elseif p(1)==p(end) && x(1)==x(end),
                            ind=(mmd.mass{i}(:,p(1))>mmd.mass{i}(:,end))|(all(mmd.mass{i}(:, 1:p(1))==mmd.mass{i}(:, end-p(end)+1:end),2)&mmd.mass{i}(:, p(1)+1)>...
                                mmd.mass{i}(:,end-p(end)));
                        end
                        mmd.mass{i}(ind,:)=mmd.mass{i}(ind, [end-p(end)+1:end, end-p(end), p(1)+2:p(1)+p(3), p(1)+1, 1:p(1)]);
                    end
                    clear ind
                end
                
                [mmd.mass{i}, ~, uind]=unique(mmd.mass{i}, 'rows'); mmd.frac{i}=accumarray(uind, mmd.frac{i});
                
                if i>1 || any(p~=mmd.p{1}) || any(x~=mmd.x{1})
                    if length(p)==1 && p==1, mmd.Properties.RowNames{i}='linear'; else, mmd.Properties.RowNames{i}=generatename(p,x);  end
                    
                end
            end
            mmd(:, 3:end)=[];
        end
    end
end

