// Matt Webster, University of Liverpool, matt@liverpool.ac.uk. April 2020.
// Works with PRISM 4.3.1.

dtmc

// Sensor states
const int start 			= 0;
const int transmit			= 1;
const int clockCycleCheck		= 2;
const int listen			= 3;
const int synchronise1			= 4;
const int synchronise2			= 5;
const int synchronise2a			= 6;
const int synchronise2b			= 7;
const int synchronise2c			= 8;
const int updateClock			= 9;
	
const int cycleLength 			= 10;
const int refractoryPeriod		= floor(cycleLength/2);  	// i.e., cycleLength/2
//const int refractoryPeriod		;  	// 
const int dutyCycle			= cycleLength;

const int sameThreshold			= 2; 	

// inboxes for s1, s2, s3
// new values sent to these sensors are stored here.
global s1InboxClock: [0..cycleLength] init 0;
global s1InboxMeta:  [0..1] init 0;
global s1InboxFull: bool init false;
global s2InboxClock: [0..cycleLength] init 0;
global s2InboxMeta:  [0..1] init 0;
global s2InboxFull: bool init false;
global s3InboxClock: [0..cycleLength] init 0;
global s3InboxMeta:  [0..1] init 0;
global s3InboxFull: bool init false;
global s4InboxClock: [0..cycleLength] init 0;
global s4InboxMeta:  [0..1] init 0;
global s4InboxFull: bool init false;

const double pClockDrift = 0;  // higher value than normal.. maybe we can reduce this
// probability that the clock has drifted by 1 time step
//const double pClockDrift = 0.000004339 * 100;  // *100 as the drift can only happen once per duty cycle, so it should be 100 times more likely to drift
//const double pClockDrift;

const int s1NextBroadcast			= 8;  	// should be randomly set

module sensor1

	s1Phase : [0..9] init transmit;
	s1Clock : [0..cycleLength] init 5;
	s1LocalMetadata : [0..1] init 1;
	s1SameCount : [0..2] init 0;

	// if local clock == next broadcast and same count < same threshold then
	// transmit local clock and local metadata value
	// restart at top of loop
	[] s1Phase=transmit & (s1Clock=s1NextBroadcast | s1Clock=(mod(s1NextBroadcast+refractoryPeriod,cycleLength))) & s1SameCount<sameThreshold -> 
		(s2InboxClock'=s1Clock) & (s2InboxMeta'=s1LocalMetadata) & (s2InboxFull'=true) &
		(s3InboxClock'=s1Clock) & (s3InboxMeta'=s1LocalMetadata) & (s3InboxFull'=true) &
		(s4InboxClock'=s1Clock) & (s4InboxMeta'=s1LocalMetadata) & (s4InboxFull'=true) & (s1SameCount'=0) & (s1Phase'=updateClock);
	[] s1Phase=transmit & !((s1Clock=s1NextBroadcast | s1Clock=(mod(s1NextBroadcast+refractoryPeriod,cycleLength))) & s1SameCount<sameThreshold) -> (s1Phase'=clockCycleCheck);

	// Algorithm: if clock ≤ cycle length then listen
	[] s1Phase=clockCycleCheck & s1Clock<cycleLength -> (s1Phase'=listen);
	[] s1Phase=clockCycleCheck & !(s1Clock<cycleLength) -> (s1Phase'=updateClock);

	// Algorithm: if A message is overheard
	// so, if the inbox clock has changed from last time, then a message must have been received, so updated the clock and the metadata and go into synchronise1 state
	[] s1Phase=listen & s1InboxFull -> (s1InboxFull'=false) & (s1Phase'=synchronise1);
	// if there is no message, then jump to the update clock state which advances the clock before going back to the start of the loop
	[] s1Phase=listen & !s1InboxFull -> (s1Phase'=updateClock);

	// if local clock > refractory period then
	// adjust local clock to average of local clock and time in message
	[] s1Phase=synchronise1 & s1Clock>=refractoryPeriod & diff<=floor(cycleLength/2) -> (s1Clock'=s1avg1) & (s1Phase'=synchronise2a);
	[] s1Phase=synchronise1 & s1Clock>=refractoryPeriod & diff>floor(cycleLength/2) -> (s1Clock'=s1avg2) & (s1Phase'=synchronise2a);
	[] s1Phase=synchronise1 & !( s1Clock>=refractoryPeriod ) -> (s1Phase'=synchronise2a); 
	// if A message is overheard AND local clock > refractory period then
	// adjust local clock to average of local clock and time in message
	// end if
	// if metadata > local metadata then local metadata := metadata
	[] s1Phase=synchronise2a & s1InboxMeta>s1LocalMetadata -> (s1LocalMetadata'=s1InboxMeta) & (s1Phase'=updateClock); 
	[] s1Phase=synchronise2a & !(s1InboxMeta>s1LocalMetadata) -> (s1Phase'=synchronise2b); 
	
	// else if metadata < local metadata AND same count < 1 then transmit data now
	[] s1Phase=synchronise2b & s1InboxMeta<s1LocalMetadata & s1SameCount<1 -> (s1Phase'=updateClock) &
											(s2InboxClock'=s1Clock) & (s2InboxMeta'=s1LocalMetadata) & (s2InboxFull'=true) &
											(s3InboxClock'=s1Clock) & (s3InboxMeta'=s1LocalMetadata) & (s3InboxFull'=true) &
											(s4InboxClock'=s1Clock) & (s4InboxMeta'=s1LocalMetadata) & (s4InboxFull'=true);
	[] s1Phase=synchronise2b & !(s1InboxMeta<s1LocalMetadata & s1SameCount<1) -> (s1Phase'=synchronise2c);
	// else if the message contains metadata == local metadata AND time in message ==localclock then same count = same count + 1
	[] s1Phase=synchronise2c & s1InboxMeta=s1LocalMetadata & s1InboxClock=s1Clock & (s1SameCount<1) -> (s1SameCount'=s1SameCount+1) & (s1Phase'=updateClock);	
	[] s1Phase=synchronise2c & s1InboxMeta=s1LocalMetadata & s1InboxClock=s1Clock & (s1SameCount>=1) -> (s1Phase'=updateClock);	
	[] s1Phase=synchronise2c & !(s1InboxMeta=s1LocalMetadata & s1InboxClock=s1Clock) -> (s1Phase'=updateClock);
	// local clock = local clock + 1
	[tick] s1Phase=updateClock & s1Clock=1 -> 		(1-pClockDrift): (s1Clock'=s1Clock+1) & (s1Phase'=transmit)
									+ pClockDrift: (s1Clock'=s1Clock+2)  & (s1Phase'=transmit);
	[tick] s1Phase=updateClock & s1Clock<cycleLength & s1Clock!=1 -> (s1Clock'=s1Clock+1) & (s1Phase'=transmit);
	[tick] s1Phase=updateClock & s1Clock=cycleLength -> (s1Clock'=0) & (s1SameCount'=0) & (s1Phase'=transmit);

endmodule

const int s2NextBroadcast			= 3;  	// should be randomly set
const int s2ClockInit  = 0;

module sensor2
	s2Phase : [0..9] init transmit;
	s2Clock : [0..cycleLength] init 5;
	s2LocalMetadata : [0..1] init 1;
	s2SameCount : [0..2] init 0;

	// if local clock == next broadcast and same count < same threshold then
	// transmit local clock and local metadata value
	// restart at top of loop
	[] s2Phase=transmit & (s2Clock=s2NextBroadcast | s2Clock=(mod(s2NextBroadcast+refractoryPeriod,cycleLength))) & s2SameCount<sameThreshold -> 
		(s1InboxClock'=s2Clock) & (s1InboxMeta'=s2LocalMetadata) & (s1InboxFull'=true) &
		(s3InboxClock'=s2Clock) & (s3InboxMeta'=s2LocalMetadata) & (s3InboxFull'=true) &
		(s4InboxClock'=s2Clock) & (s4InboxMeta'=s2LocalMetadata) & (s4InboxFull'=true) & (s2SameCount'=0) & (s2Phase'=updateClock);
	[] s2Phase=transmit & !((s2Clock=s2NextBroadcast | s2Clock=(mod(s2NextBroadcast+refractoryPeriod,cycleLength))) & s2SameCount<sameThreshold) -> (s2Phase'=clockCycleCheck);

	// Algorithm: if clock ≤ cycle length then listen
	[] s2Phase=clockCycleCheck & s2Clock<cycleLength -> (s2Phase'=listen);
	[] s2Phase=clockCycleCheck & !(s2Clock<cycleLength) -> (s2Phase'=updateClock);

	// Algorithm: if A message is overheard
	// so, if the inbox clock has changed from last time, then a message must have been received, so updated the clock and the metadata and go into synchronise1 state
	[] s2Phase=listen & s2InboxFull -> (s2InboxFull'=false) & (s2Phase'=synchronise1);
	// if there is no message, then jump to the update clock state which advances the clock before going back to the start of the loop
	[] s2Phase=listen & !s2InboxFull -> (s2Phase'=updateClock);

	// if local clock > refractory period then
	// adjust local clock to average of local clock and time in message
	[] s2Phase=synchronise1 & s2Clock>=refractoryPeriod & diff<=floor(cycleLength/2) -> (s2Clock'=s2avg1) & (s2Phase'=synchronise2a);
	[] s2Phase=synchronise1 & s2Clock>=refractoryPeriod & diff>floor(cycleLength/2) -> (s2Clock'=s2avg2) & (s2Phase'=synchronise2a);
	[] s2Phase=synchronise1 & !( s2Clock>=refractoryPeriod ) -> (s2Phase'=synchronise2a); 
	// if A message is overheard AND local clock > refractory period then
	// adjust local clock to average of local clock and time in message
	// end if
	// if metadata > local metadata then local metadata := metadata
	[] s2Phase=synchronise2a & s2InboxMeta>s2LocalMetadata -> (s2LocalMetadata'=s2InboxMeta) & (s2Phase'=updateClock); 
	[] s2Phase=synchronise2a & !(s2InboxMeta>s2LocalMetadata) -> (s2Phase'=synchronise2b); 
	
	// else if metadata < local metadata AND same count < 1 then transmit data now
	[] s2Phase=synchronise2b & s2InboxMeta<s2LocalMetadata & s2SameCount<1 -> (s2Phase'=updateClock) &
											(s1InboxClock'=s2Clock) & (s1InboxMeta'=s2LocalMetadata) & (s1InboxFull'=true) &
											(s3InboxClock'=s2Clock) & (s3InboxMeta'=s2LocalMetadata) & (s3InboxFull'=true) &
											(s4InboxClock'=s2Clock) & (s4InboxMeta'=s2LocalMetadata) & (s4InboxFull'=true);
	[] s2Phase=synchronise2b & !(s2InboxMeta<s2LocalMetadata & s2SameCount<1) -> (s2Phase'=synchronise2c);
	// else if the message contains metadata == local metadata AND time in message ==localclock then same count = same count + 1
	[] s2Phase=synchronise2c & s2InboxMeta=s2LocalMetadata & s2InboxClock=s2Clock & (s2SameCount<1) -> (s2SameCount'=s2SameCount+1) & (s2Phase'=updateClock);	
	[] s2Phase=synchronise2c & s2InboxMeta=s2LocalMetadata & s2InboxClock=s2Clock & (s2SameCount>=1) -> (s2Phase'=updateClock);	
	[] s2Phase=synchronise2c & !(s2InboxMeta=s2LocalMetadata & s2InboxClock=s2Clock) -> (s2Phase'=updateClock);
	// local clock = local clock + 1
	[tick] s2Phase=updateClock & s2Clock=1 -> 		(1-pClockDrift): (s2Clock'=s2Clock+1) & (s2Phase'=transmit)
									+ pClockDrift: (s2Clock'=s2Clock+2)  & (s2Phase'=transmit);
	[tick] s2Phase=updateClock & s2Clock<cycleLength & s2Clock!=1 -> (s2Clock'=s2Clock+1) & (s2Phase'=transmit);
	[tick] s2Phase=updateClock & s2Clock=cycleLength -> (s2Clock'=0) & (s2SameCount'=0) & (s2Phase'=transmit);


endmodule

const int s3NextBroadcast			= 6;  	// should be randomly set

module sensor3

	s3Phase : [0..9] init transmit;
	s3Clock : [0..cycleLength] init 7;
	s3LocalMetadata : [0..1] init 1;
	s3SameCount : [0..2] init 0;


	// if local clock == next broadcast and same count < same threshold then
	// transmit local clock and local metadata value
	// restart at top of loop
	[] s3Phase=transmit & (s3Clock=s3NextBroadcast | s3Clock=(mod(s3NextBroadcast+refractoryPeriod,cycleLength))) & s3SameCount<sameThreshold -> 
		(s2InboxClock'=s3Clock) & (s2InboxMeta'=s3LocalMetadata) & (s2InboxFull'=true) &
		(s1InboxClock'=s3Clock) & (s1InboxMeta'=s3LocalMetadata) & (s1InboxFull'=true) & (s3SameCount'=0) & (s3Phase'=updateClock);
	[] s3Phase=transmit & !((s3Clock=s3NextBroadcast | s3Clock=(mod(s3NextBroadcast+refractoryPeriod,cycleLength))) & s3SameCount<sameThreshold) -> (s3Phase'=clockCycleCheck);


	// Algorithm: if clock ≤ cycle length then listen
	[] s3Phase=clockCycleCheck & s3Clock<cycleLength -> (s3Phase'=listen);
	[] s3Phase=clockCycleCheck & !(s3Clock<cycleLength) -> (s3Phase'=updateClock);

	// Algorithm: if A message is overheard
	// so, if the inbox clock has changed from last time, then a message must have been received, so updated the clock and the metadata and go into synchronise1 state
	[] s3Phase=listen & s3InboxFull -> (s3InboxFull'=false) & (s3Phase'=synchronise1);
	// likewise for the inbox metadata
	// if there is no message, then jump to the update clock state which advances the clock before going back to the start of the loop
	[] s3Phase=listen & !s3InboxFull -> (s3Phase'=updateClock);

	// if local clock > refractory period then
	// adjust local clock to average of local clock and time in message
	[] s3Phase=synchronise1 & s3Clock>=refractoryPeriod & diff<=floor(cycleLength/2) -> (s3Clock'=s3avg1) & (s3Phase'=synchronise2a);
	[] s3Phase=synchronise1 & s3Clock>=refractoryPeriod & diff>floor(cycleLength/2) -> (s3Clock'=s3avg2) & (s3Phase'=synchronise2a);
	[] s3Phase=synchronise1 & !( s3Clock>=refractoryPeriod ) -> (s3Phase'=synchronise2a); 
	// if A message is overheard AND local clock > refractory period then
	// adjust local clock to average of local clock and time in message
	// end if
	// if metadata > local metadata then local metadata := metadata
	[] s3Phase=synchronise2a & s3InboxMeta>s3LocalMetadata -> (s3LocalMetadata'=s3InboxMeta) & (s3Phase'=updateClock); 
	[] s3Phase=synchronise2a & !(s3InboxMeta>s3LocalMetadata) -> (s3Phase'=synchronise2b); 
	
	// else if metadata < local metadata AND same count < 1 then transmit data now
	[] s3Phase=synchronise2b & s3InboxMeta<s3LocalMetadata & s3SameCount<1 -> (s3Phase'=updateClock) &
											(s2InboxClock'=s3Clock) & (s2InboxMeta'=s3LocalMetadata) & (s2InboxFull'=true) &
											(s1InboxClock'=s3Clock) & (s1InboxMeta'=s3LocalMetadata) & (s1InboxFull'=true);
	[] s3Phase=synchronise2b & !(s3InboxMeta<s3LocalMetadata & s3SameCount<1) -> (s3Phase'=synchronise2c);
	// else if the message contains metadata == local metadata AND time in message ==localclock then same count = same count + 1
	[] s3Phase=synchronise2c & s3InboxMeta=s3LocalMetadata & s3InboxClock=s3Clock & (s3SameCount<1) -> (s3SameCount'=s3SameCount+1) & (s3Phase'=updateClock);	
	[] s3Phase=synchronise2c & s3InboxMeta=s3LocalMetadata & s3InboxClock=s3Clock & (s3SameCount>=1) -> (s3Phase'=updateClock);	
	[] s3Phase=synchronise2c & !(s3InboxMeta=s3LocalMetadata & s3InboxClock=s3Clock) -> (s3Phase'=updateClock);
	// local clock = local clock + 1
	[tick] s3Phase=updateClock & s3Clock=1 -> 		(1-pClockDrift): (s3Clock'=s3Clock+1) & (s3Phase'=transmit)
									+ pClockDrift: (s3Clock'=s3Clock+2)  & (s3Phase'=transmit);
	[tick] s3Phase=updateClock & s3Clock<cycleLength & s3Clock!=1 -> (s3Clock'=s3Clock+1) & (s3Phase'=transmit);
	[tick] s3Phase=updateClock & s3Clock=cycleLength -> (s3Clock'=0) & (s3SameCount'=0) & (s3Phase'=transmit);

endmodule

const int s4NextBroadcast			= 2;  	// should be randomly set

module sensor4
	s4Phase : [0..9] init transmit;
	s4Clock : [0..cycleLength] init 9;
	s4LocalMetadata : [0..1] init 1;
	s4SameCount : [0..2] init 0;

	// if local clock == next broadcast and same count < same threshold then
	// transmit local clock and local metadata value
	// restart at top of loop
	[] s4Phase=transmit & (s4Clock=s4NextBroadcast | s4Clock=(mod(s4NextBroadcast+refractoryPeriod,cycleLength))) & s4SameCount<sameThreshold -> 
		(s2InboxClock'=s4Clock) & (s2InboxMeta'=s4LocalMetadata) & (s2InboxFull'=true) &
		(s3InboxClock'=s4Clock) & (s3InboxMeta'=s4LocalMetadata) & (s3InboxFull'=true) &
		(s1InboxClock'=s4Clock) & (s1InboxMeta'=s4LocalMetadata) & (s1InboxFull'=true) & (s4SameCount'=0) & (s4Phase'=updateClock);
	[] s4Phase=transmit & !((s4Clock=s4NextBroadcast | s4Clock=(mod(s4NextBroadcast+refractoryPeriod,cycleLength))) & s4SameCount<sameThreshold) -> (s4Phase'=clockCycleCheck);

	// Algorithm: if clock ≤ cycle length then listen
	[] s4Phase=clockCycleCheck & s4Clock<cycleLength -> (s4Phase'=listen);
	[] s4Phase=clockCycleCheck & !(s4Clock<cycleLength) -> (s4Phase'=updateClock);

	// Algorithm: if A message is overheard
	// so, if the inbox clock has changed from last time, then a message must have been received, so updated the clock and the metadata and go into synchronise1 state
	[] s4Phase=listen & s4InboxFull -> (s4InboxFull'=false) & (s4Phase'=synchronise1);
	// if there is no message, then jump to the update clock state which advances the clock before going back to the start of the loop
	[] s4Phase=listen & !s4InboxFull -> (s4Phase'=updateClock);

	// if local clock > refractory period then
	// adjust local clock to average of local clock and time in message
	[] s4Phase=synchronise1 & s4Clock>=refractoryPeriod & diff<=floor(cycleLength/2) -> (s4Clock'=s4avg1) & (s4Phase'=synchronise2a);
	[] s4Phase=synchronise1 & s4Clock>=refractoryPeriod & diff>floor(cycleLength/2) -> (s4Clock'=s4avg2) & (s4Phase'=synchronise2a);
	[] s4Phase=synchronise1 & !( s4Clock>=refractoryPeriod ) -> (s4Phase'=synchronise2a); 
	// if A message is overheard AND local clock > refractory period then
	// adjust local clock to average of local clock and time in message
	// end if
	// if metadata > local metadata then local metadata := metadata
	[] s4Phase=synchronise2a & s4InboxMeta>s4LocalMetadata -> (s4LocalMetadata'=s4InboxMeta) & (s4Phase'=updateClock); 
	[] s4Phase=synchronise2a & !(s4InboxMeta>s4LocalMetadata) -> (s4Phase'=synchronise2b); 
	
	// else if metadata < local metadata AND same count < 1 then transmit data now
	[] s4Phase=synchronise2b & s4InboxMeta<s4LocalMetadata & s4SameCount<1 -> (s4Phase'=updateClock) &
											(s2InboxClock'=s4Clock) & (s2InboxMeta'=s4LocalMetadata) & (s2InboxFull'=true) &
											(s3InboxClock'=s4Clock) & (s3InboxMeta'=s4LocalMetadata) & (s3InboxFull'=true) &
											(s1InboxClock'=s4Clock) & (s1InboxMeta'=s4LocalMetadata) & (s1InboxFull'=true);
	[] s4Phase=synchronise2b & !(s4InboxMeta<s4LocalMetadata & s4SameCount<1) -> (s4Phase'=synchronise2c);
	// else if the message contains metadata == local metadata AND time in message ==localclock then same count = same count + 1
	[] s4Phase=synchronise2c & s4InboxMeta=s4LocalMetadata & s4InboxClock=s4Clock & (s4SameCount<1) -> (s4SameCount'=s4SameCount+1) & (s4Phase'=updateClock);	
	[] s4Phase=synchronise2c & s4InboxMeta=s4LocalMetadata & s4InboxClock=s4Clock & (s4SameCount>=1) -> (s4Phase'=updateClock);	
	[] s4Phase=synchronise2c & !(s4InboxMeta=s4LocalMetadata & s4InboxClock=s4Clock) -> (s4Phase'=updateClock);
	// local clock = local clock + 1
	[tick] s4Phase=updateClock & s4Clock=1 -> 		(1-pClockDrift): (s4Clock'=s4Clock+1) & (s4Phase'=transmit)
									+ pClockDrift: (s4Clock'=s4Clock+2)  & (s4Phase'=transmit);
	[tick] s4Phase=updateClock & s4Clock<cycleLength & s4Clock!=1 -> (s4Clock'=s4Clock+1) & (s4Phase'=transmit);
	[tick] s4Phase=updateClock & s4Clock=cycleLength -> (s4Clock'=0) & (s4SameCount'=0) & (s4Phase'=transmit);

endmodule

formula diff = max(s1Clock,s2Clock)-min(s1Clock,s2Clock);

// averages for s1
formula s1avg1 = ceil((s1Clock+s1InboxClock)/2);
formula s1avg2 = mod(ceil((s1Clock+s1InboxClock+cycleLength)/2),cycleLength);
// averages for s2
formula s2avg1 = ceil((s2Clock+s2InboxClock)/2);
formula s2avg2 = mod(ceil((s2Clock+s2InboxClock+cycleLength)/2),cycleLength);
// averages for s3
formula s3avg1 = ceil((s3Clock+s3InboxClock)/2);
formula s3avg2 = mod(ceil((s3Clock+s3InboxClock+cycleLength)/2),cycleLength);
// averages for s4
formula s4avg1 = ceil((s4Clock+s4InboxClock)/2);
formula s4avg2 = mod(ceil((s4Clock+s4InboxClock+cycleLength)/2),cycleLength);

// END OF FILE
