// Matt Webster, University of Liverpool, matt@liverpool.ac.uk. April 2020.
// Works with PRISM 4.3.1.

dtmc

// Sensor states
const int start 			= 0;
const int transmit			= 1;
const int clockCycleCheck		= 2;
const int listen			= 3;
const int synchronise1			= 4;
const int synchronise2			= 5;
const int synchronise2a			= 6;
const int synchronise2b			= 7;
const int synchronise2c			= 8;
const int updateClock			= 9;

const int cycleLength 			= 100;
const int refractoryPeriod		= floor(cycleLength/2);  	// i.e., cycleLength/2
//const int refractoryPeriod		;  	// variable for experiments
const int dutyCycle			= cycleLength;
const int sameThreshold			= 2; 	

// inboxes for s1, s2
// new values sent to these sensors are stored here.
global s1InboxClock: [0..cycleLength] init 0;
global s1InboxMeta:  [0..1] init 0;
global s1InboxFull: bool init false;
global s2InboxClock: [0..cycleLength] init 0;
global s2InboxMeta:  [0..1] init 0;
global s2InboxFull: bool init false;

// probability that the clock has drifted by 1 time step
const double pClockDrift = 0;  
//const double pClockDrift = 0.000004339 * 100;  // *100 as the drift can only happen once per duty cycle, so it should be 100 times more likely to drift
//const double pClockDrift;		// variable for experiments

const int s1NextBroadcast			= 10;  	// should be randomly set

module sensor1

	s1Phase : [0..9] init transmit;
	s1Clock : [0..cycleLength] init 5;
	s1LocalMetadata : [0..1] init 1;
	s1SameCount : [0..2] init 0;

	// if local clock == next broadcast and same count < same threshold then
	// transmit local clock and local metadata value
	// restart at top of loop

	[] s1Phase=transmit & (s1Clock=s1NextBroadcast | s1Clock=(mod(s1NextBroadcast+refractoryPeriod,cycleLength))) & s1SameCount<sameThreshold -> (s2InboxClock'=s1Clock) & (s2InboxMeta'=s1LocalMetadata) & (s2InboxFull'=true) & (s1SameCount'=0) & (s1Phase'=updateClock);
	[] s1Phase=transmit & !((s1Clock=s1NextBroadcast | s1Clock=(mod(s1NextBroadcast+refractoryPeriod,cycleLength))) & s1SameCount<sameThreshold) -> (s1Phase'=clockCycleCheck);

	// Algorithm: if clock ≤ cycle length then listen
	[] s1Phase=clockCycleCheck & s1Clock<cycleLength -> (s1Phase'=listen);
	[] s1Phase=clockCycleCheck & !(s1Clock<cycleLength) -> (s1Phase'=updateClock);

	// Algorithm: if A message is overheard
	// so, if the inbox clock has changed from last time, then a message must have been received, so updated the clock and the metadata and go into synchronise1 state
	[] s1Phase=listen & s1InboxFull -> (s1InboxFull'=false) & (s1Phase'=synchronise1);
	// if there is no message, then jump to the update clock state which advances the clock before going back to the start of the loop
	[] s1Phase=listen & !s1InboxFull -> (s1Phase'=updateClock);

	// if local clock > refractory period then
	// adjust local clock to average of local clock and time in message
	[] s1Phase=synchronise1 & s1Clock>=refractoryPeriod & diff<=floor(cycleLength/2) -> (s1Clock'=s1avg1) & (s1Phase'=synchronise2a);
	[] s1Phase=synchronise1 & s1Clock>=refractoryPeriod & diff>floor(cycleLength/2) -> (s1Clock'=s1avg2) & (s1Phase'=synchronise2a);
	[] s1Phase=synchronise1 & !( s1Clock>=refractoryPeriod ) -> (s1Phase'=synchronise2a); 
	// if A message is overheard AND local clock > refractory period then
	// adjust local clock to average of local clock and time in message
	// end if
	// if metadata > local metadata then local metadata := metadata
	[] s1Phase=synchronise2a & s1InboxMeta>s1LocalMetadata -> (s1LocalMetadata'=s1InboxMeta) & (s1Phase'=updateClock); 
	[] s1Phase=synchronise2a & !(s1InboxMeta>s1LocalMetadata) -> (s1Phase'=synchronise2b); 
	
	// else if metadata < local metadata AND same count < 1 then transmit data now
	[] s1Phase=synchronise2b & s1InboxMeta<s1LocalMetadata & s1SameCount<1 -> (s2InboxClock'=s1Clock) & (s2InboxMeta'=s1LocalMetadata) & (s2InboxFull'=true) & (s1Phase'=updateClock);
	[] s1Phase=synchronise2b & !(s1InboxMeta<s1LocalMetadata & s1SameCount<1) -> (s1Phase'=synchronise2c);
	// else if the message contains metadata == local metadata AND time in message ==localclock then same count = same count + 1
	[] s1Phase=synchronise2c & s1InboxMeta=s1LocalMetadata & s1InboxClock=s1Clock & (s1SameCount<1) -> (s1SameCount'=s1SameCount+1) & (s1Phase'=updateClock);	
	[] s1Phase=synchronise2c & s1InboxMeta=s1LocalMetadata & s1InboxClock=s1Clock & (s1SameCount>=1) -> (s1Phase'=updateClock);	
	[] s1Phase=synchronise2c & !(s1InboxMeta=s1LocalMetadata & s1InboxClock=s1Clock) -> (s1Phase'=updateClock);
	// local clock = local clock + 1
	[tick] s1Phase=updateClock & s1Clock=1 -> 		(1-pClockDrift): (s1Clock'=s1Clock+1) & (s1Phase'=transmit)
									+ pClockDrift: (s1Clock'=s1Clock+2)  & (s1Phase'=transmit);
	[tick] s1Phase=updateClock & s1Clock<cycleLength & s1Clock!=1 -> (s1Clock'=s1Clock+1) & (s1Phase'=transmit);
	[tick] s1Phase=updateClock & s1Clock=cycleLength -> (s1Clock'=0) & (s1SameCount'=0) & (s1Phase'=transmit);

endmodule

const int s2NextBroadcast			= 3;  	// should be randomly set
const int s2ClockInit  = 0;

module sensor2
	s2Phase : [0..9] init transmit;
	s2Clock : [0..cycleLength] init s2ClockInit;
	s2LocalMetadata : [0..1] init 1;
	s2SameCount : [0..2] init 0;


	// if local clock == next broadcast and same count < same threshold then
	// transmit local clock and local metadata value
	// restart at top of loop
	[] s2Phase=transmit & (s2Clock=s2NextBroadcast | s2Clock=(mod(s2NextBroadcast+refractoryPeriod,cycleLength))) & s2SameCount<sameThreshold -> (s1InboxClock'=s2Clock) & (s1InboxMeta'=s2LocalMetadata) & (s1InboxFull'=true) & (s2SameCount'=0) & (s2Phase'=updateClock);
	//[] s1Phase=transmit & (s1Clock=s1NextBroadcast | s1Clock=(s1NextBroadcast+refractoryPeriod)) & s1SameCount<sameThreshold -> (s2InboxClock'=s1Clock) & (s2InboxMeta'=s1LocalMetadata) & (s2InboxFull'=true) & (s1SameCount'=0) & (s1Phase'=updateClock);
	[] s2Phase=transmit & !((s2Clock=s2NextBroadcast | s2Clock=(mod(s2NextBroadcast+refractoryPeriod,cycleLength))) & s2SameCount<sameThreshold) -> (s2Phase'=clockCycleCheck);

	
	// Algorithm: if clock ≤ cycle length then listen
	[] s2Phase=clockCycleCheck & s2Clock<cycleLength -> (s2Phase'=listen);
	[] s2Phase=clockCycleCheck & !(s2Clock<cycleLength) -> (s2Phase'=updateClock);

	// Algorithm: if A message is overheard
	// so, if the inbox clock has changed from last time, then a message must have been received, so updated the clock and the metadata and go into synchronise1 state
	[] s2Phase=listen & s2InboxFull -> (s2InboxFull'=false) & (s2Phase'=synchronise1);
	// likewise for the inbox metadata
	//[] s2Phase=listen & s2InboxFull ->(s2Phase'=synchronise1);
	// if there is no message, then jump to the update clock state which advances the clock before going back to the start of the loop
	[] s2Phase=listen & !s2InboxFull -> (s2Phase'=updateClock);

	// if local clock > refractory period then
	// adjust local clock to average of local clock and time in message
	[] s2Phase=synchronise1 & s2Clock>=refractoryPeriod & diff<=floor(cycleLength/2) -> (s2Clock'=s2avg1) & (s2Phase'=synchronise2a);
	[] s2Phase=synchronise1 & s2Clock>=refractoryPeriod & diff>floor(cycleLength/2) -> (s2Clock'=s2avg2) & (s2Phase'=synchronise2a);
	[] s2Phase=synchronise1 & !( s2Clock>=refractoryPeriod ) -> (s2Phase'=synchronise2a);
	// if A message is overheard AND local clock > refractory period then
	// adjust local clock to average of local clock and time in message
	// end if
	// if metadata > local metadata then local metadata := metadata
	[] s2Phase=synchronise2a & s2InboxMeta>s2LocalMetadata -> (s2LocalMetadata'=s2InboxMeta) & (s2Phase'=updateClock); 
	[] s2Phase=synchronise2a & !(s2InboxMeta>s2LocalMetadata) -> (s2Phase'=synchronise2b); 
	
	// else if metadata < local metadata AND same count < 1 then transmit data now
	[] s2Phase=synchronise2b & s2InboxMeta<s2LocalMetadata & s2SameCount<1 -> (s1InboxClock'=s2Clock) & (s1InboxMeta'=s2LocalMetadata) & (s1InboxFull'=true) & (s2Phase'=updateClock);
	[] s2Phase=synchronise2b & !(s2InboxMeta<s2LocalMetadata & s2SameCount<1) -> (s2Phase'=synchronise2c);
	// else if the message contains metadata == local metadata AND time in message ==localclock then same count = same count + 1
	[] s2Phase=synchronise2c & s2InboxMeta=s2LocalMetadata & s2InboxClock=s2Clock & (s2SameCount<1) -> (s2SameCount'=s2SameCount+1) & (s2Phase'=updateClock);	
	[] s2Phase=synchronise2c & s2InboxMeta=s2LocalMetadata & s2InboxClock=s2Clock & (s2SameCount>=1) -> (s2Phase'=updateClock);	
	[] s2Phase=synchronise2c & !(s2InboxMeta=s2LocalMetadata & s2InboxClock=s2Clock) -> (s2Phase'=updateClock);
	// local clock = local clock + 1
	[tick] s2Phase=updateClock & s2Clock=cycleLength -> (s2Clock'=0) & (s2SameCount'=0) & (s2Phase'=transmit);
	[tick] s2Phase=updateClock & s2Clock<cycleLength -> (s2Clock'=s2Clock+1) & (s2Phase'=transmit);

endmodule

formula diff = max(s1Clock,s2Clock)-min(s1Clock,s2Clock);

// averages for s1
formula s1avg1 = ceil((s1Clock+s1InboxClock)/2);
formula s1avg2 = mod(ceil((s1Clock+s1InboxClock+cycleLength)/2),cycleLength);
// averages for s2
formula s2avg1 = ceil((s2Clock+s2InboxClock)/2);
formula s2avg2 = mod(ceil((s2Clock+s2InboxClock+cycleLength)/2),cycleLength);

// END OF FILE
